/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __WIN32__
#include <unistd.h>
#else
#include "win32.h"
#endif
#include <errno.h>
#include "basescheduler_decls.h"
#include "callback.h"
#include "currentthread.h"
#include "timeval.h"
#include "globalqueue.h"

BaseScheduler::BaseScheduler(BlockingQueue<Microthread*> *q)
		: Microthread(q ? q : &getglobalqueue()) {}

void BaseScheduler::scheduleRead(int fd) {
	this->addReadCallback(*new ScheduleCallback(getcurrentthread()), fd);
}

void BaseScheduler::scheduleWrite(int fd) {
	this->addWriteCallback(*new ScheduleCallback(getcurrentthread()), fd);
}

void BaseScheduler::scheduleWait(const timeval &tv) {
	this->addWaitCallback(*new ScheduleCallback(getcurrentthread()), tv);
}

ssize_t BaseScheduler::read(int fd, void *buf, size_t count) {
	for(;;) {
		this->scheduleRead(fd);
		getcurrentthread().delayme();
#ifndef __WIN32__
		const ssize_t r(::read(fd, buf, count));
#else
		const ssize_t r(::recv(fd, (char*)buf, count, 0));
#endif
		if((r >= 0) || ((errno != EAGAIN) && (errno != EINTR)))
			return r;
	}
}

ssize_t BaseScheduler::write(int fd, const void *buf, size_t count) {
	for(;;) {
		this->scheduleWrite(fd);
		getcurrentthread().delayme();
#ifndef __WIN32__
		const ssize_t r(::write(fd, buf, count));
#else
		const ssize_t r(::send(fd, (const char*)buf, count, 0));
#endif
		if((r >= 0) || ((errno != EAGAIN) && (errno != EINTR)))
			return r;
	}
}

int BaseScheduler::accept(int sockfd, struct sockaddr *addr, socklen_t
		*addrlen) {
	for(;;) {
		this->scheduleRead(sockfd);
		getcurrentthread().delayme();
		const int r(::accept(sockfd, addr, addrlen));
#ifndef __WIN32__
		if((r >= 0) || ((errno != EAGAIN) && (errno != EWOULDBLOCK) &&
					(errno != EINTR)))
#else
		if((r >= 0) || ((errno != EAGAIN) && (errno != EINTR)))
#endif
			return r;
	}
}

int BaseScheduler::connect(int sockfd, const struct sockaddr *serv_addr,
		socklen_t addrlen) {
	for(;;) {
		this->scheduleWrite(sockfd);
		getcurrentthread().delayme();
		const int r(::connect(sockfd, serv_addr, addrlen));
#ifndef __WIN32__
		if((r >= 0) || ((errno != EALREADY) && (errno != EINPROGRESS) &&
					(errno != EINTR)))
#else
		if((r >= 0) || (errno != EINTR))
#endif
			return r;
	}
}

unsigned int BaseScheduler::sleep(unsigned int seconds) {
	Timeval tv(TimevalNow);
	tv += (time_t)seconds;
	this->scheduleWait(tv);
	getcurrentthread().delayme();
	return 0;
}
