/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_BASESCHEDULER_DECLS_H
#define LIBMUTH_BASESCHEDULER_DECLS_H

#include <sys/types.h>
#ifndef __WIN32__
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#else
#include "win32.h"
#endif
#include "blockingqueue_decls.h"
#include "callback_decls.h"
#include "channel_decls.h"
#include "microthread.h"

/**
 * This class defines the interface for all scheduler implementations.
 */
class BaseScheduler : public Microthread {
	protected:
		/**
		 * BaseScheduler constructor. This class is not instanciateable.
		 * @param q is the queue passed to the Microthread constructor
		 */
		BaseScheduler(BlockingQueue<Microthread*> *q=NULL);
	public:
		/**
		 * Tells this Scheduler to run cb when fd is readable.
		 * This method must be overridden by derived classes. The
		 * Callback must not throw an exception when run.
		 * @param cb is the Callback to be run
		 * @param fd is the filedescriptor to be monitored
		 */
		virtual void addReadCallback(Callback &cb, int fd)=0;
		/**
		 * Tells this Scheduler to run cb when fd is writeable.
		 * This method must be overridden by derived classes. The
		 * Callback must not throw an exception when run.
		 * @param cb is the Callback to be run
		 * @param fd is the filedescriptor to be monitored
		 */
		virtual void addWriteCallback(Callback &cb, int fd)=0;
		/**
		 * Tells this Scheduler to run cb when tv is reached.
		 * This method must be overridden by derived classes. The
		 * Callback must not throw an exception when run.
		 * @param cb is the Callback to be run
		 * @param tv is the absolute time to be reached
		 */
		virtual void addWaitCallback(Callback &cb,
				const timeval &tv)=0;
		/**
		 * Continues the current thread when fd is readable.
		 * @param fd is the filedescriptor to be monitored
		 */
		void scheduleRead(int fd);
		/**
		 * Continues the current thread when fd is writable.
		 * @param fd is the filedescriptor to be monitored
		 */
		void scheduleWrite(int fd);
		/**
		 * Continues the current thread when tv is reached.
		 * @param tv is the absolute time to be reached
		 */
		void scheduleWait(const timeval &tv);
		/**
		 * Sends value to chan when fd is readable.
		 * @param chan is the Channel to send value on
		 * @param value is the object to send on chan
		 * @param fd is the filedescriptor to be monitored
		 */
		template<class T> void notifyRead(Channel<T> &chan,
				const T &value, int fd);
		/**
		 * Sends value to chan when fd is writable.
		 * @param chan is the Channel to send value on
		 * @param value is the object to send on chan
		 * @param fd is the filedescriptor to be monitored
		 */
		template<class T> void notifyWrite(Channel<T> &chan,
				const T &value, int fd);
		/**
		 * Sends value to chan when tv is reached.
		 * @param chan is the Channel to send value on
		 * @param value is the object to send on chan
		 * @param tv is the absolute time to be reached
		 */
		template<class T> void notifyWait(Channel<T> &chan,
				const T &value, const timeval &tv);
		virtual void run()=0;
		/**
		 * This method behaves like the function described in man 2
		 * read. The only difference is that it first delays the
		 * current microthread and continues execution when the
		 * filedescriptor is readable so that the read happens
		 * non-blocking.
		 * @param fd is passed to read and scheduled for reading before
		 * @param buf is passed to read
		 * @param count is passed to read
		 * @returns the value read returns
		 * @throws ThreadExitException
		 */
		ssize_t read(int fd, void *buf, size_t count);
		/**
		 * This method behaves like the function described in man 2
		 * write. The only difference is that it first delays the
		 * current microthread and continues exaction when the
		 * filedescriptor is writable so that the write happens
		 * non-blocking.
		 * @param fd is passed to write and scheduled for writing before
		 * @param buf is passed to write
		 * @param count is passed to write
		 * @returns the value write returns
		 * @throws ThreadExitException
		 */
		ssize_t write(int fd, const void *buf, size_t count);
		/**
		 * This method behaves like the function described in man 2
		 * accept. The only difference is that it first delays the
		 * current microthread and continues execution when the
		 * filedescriptor is readable so that the accept happens
		 * non-blocking.
		 * @param sockfd is passed to accept and scheduled for reading
		 *		 before
		 * @param addr is passed to accept
		 * @param addrlen is passed to accept
		 * @returns the value accept returns
		 * @throws ThreadExitException
		 */
		int accept(int sockfd, struct sockaddr *addr,
				socklen_t *addrlen);
		/**
		 * This method behaves like the function described in man 2
		 * connect. The only difference is that it first delays the
		 * current microthread and continues execution when the
		 * filedescriptor is writable so the connect happens
		 * non-blocking.
		 * @param sockfd is passed to connect an scheduled for writing
		 * 		 before
		 * @param serv_addr is passed to connect
		 * @param addrlen is passed to connect
		 * @returns the value connect returns
		 * @throws ThreadExitException
		 */
		int connect(int sockfd, const struct sockaddr *serv_addr,
				socklen_t addrlen);
		/**
		 * This method behaves like the function described in man 3
		 * sleep. It does not put all threads into sleep, only the one
		 * currently running is interrupted. This sleep call is not
		 * interruptible by signals.
		 * @param seconds is the time in seconds to sleep
		 * @returns 0
		 * @throws ThreadExitException
		 */
		unsigned int sleep(unsigned int seconds);
};

#endif
