/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_BLOCKINGQUEUE_H
#define LIBMUTH_BLOCKINGQUEUE_H

#include "blockingqueue_decls.h"
#include "lock.h"
#include "condition.h"

template<class C> inline BlockingQueue<C>::BlockingQueue() : notempty(lock) {
}

template<class C> inline bool BlockingQueue<C>::isempty() const {
	/* We need to lock the queue before asking it whether it is empty. */
	WITHACQUIRED(this->lock);
	return this->queue.empty();
}

template<class C> inline void BlockingQueue<C>::put(const C &item) {
	/* The queue has to be locked for all operations. */
	WITHACQUIRED(this->lock);
	/* Check whether the queue is empty. We only need to notify other
	 * threads if this is the case. */
	const bool wasempty(this->queue.empty());
	this->queue.push(item);
	if(wasempty)
		this->notempty.notify();
}

template<class C> inline C BlockingQueue<C>::get() {
	WITHACQUIRED(this->lock);
	/* If the queue is empty wait. Waiting means unlocking it and locking
	 * it when another thread has invoked .notify() and .release()d the
	 * lock. */
	while(this->queue.empty())
		this->notempty.wait();
	/* Get and remove the first element. */
	C t(this->queue.front());
	this->queue.pop();
	return t;
}

#endif
