/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_BLOCKINGQUEUE_DECLS_H
#define LIBMUTH_BLOCKINGQUEUE_DECLS_H

#include <queue>
#include "lock.h"
#include "condition.h"

/**
 * A queue to put and get (blocking) items.
 * Items can be put using .put(item) which is nonblocking. They can be taken
 * out with blockingly with .get() if the queue is empty (when .isempty()
 * returns true). It will return when another thread .put()s an item.
 */

template<class C> class BlockingQueue {
	private:
		mutable Lock lock;
		mutable Condition notempty;
		std::queue<C> queue;
	public:
		/**
		 * BlockingQueue constructor.
		 */
		BlockingQueue();
		/**
		 * @return true if the queue is empty
		 */
		bool isempty() const;
		/**
		 * Puts the given item in the queue.
		 * This will use the item's copy operator at least once. The
		 * operation is nonblocking.
		 * @param item is the item to be put.
		 */
		void put(const C &item);
		/**
		 * Gets an item from the queue.
		 * If the queue is empty, this operation will block the current
		 * thread until another thread put()s an item in the queue. If
		 * another thread also waits to get() something one of them will
		 * get the object being put(). Which one of the waiting thread
		 * actually gets the object is unspecified.
		 * @return the item that was on top of the queue
		 */
		C get();
};

#endif
