/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_CALLBACK_DECLS_H
#define LIBMUTH_CALLBACK_DECLS_H

#include <list>
#include "microthread.h"
#include "channel_decls.h"
#include "suspender_decls.h"

/**
 * This class defines an interface for all callback classes to be used with the
 * scheduler. Subclasses must override the run() method. You may also override
 * the iscompleted() method to make it possible to cancel your callback.
 */
class Callback {
	public:
		virtual ~Callback();
		/**
		 * This method takes action when the Callback is to be run.
		 * Derived classes must override this method.
		 */
		virtual void run()=0;
		/**
		 * This method returns true for canceled callbacks. It is not
		 * invoked after the run() method. This method may be
		 * overridden.
		 * @return false
		 */
		virtual bool iscompleted();
};

/**
 * This Callback schedules the Microthread given to the constructor when run.
 */
class ScheduleCallback : public Callback {
	private:
		const WeakReference<Microthread> microthread;
	public:
		/**
		 * @param m is the Microthread to be run
		 */
		ScheduleCallback(Microthread &m);
		void run();
};


/**
 * This Callback sends a given object on a given Channel when run.
 */
template<class T> class ChannelCallback : public Callback {
	private:
		Channel<T> &channel;
		T value;
	public:
		/**
		 * ChannelCallback constructor.
		 * @param c is the Channel that should receive the object
		 * @param v is the object to be sent on the Channel
		 * @see Channel
		 */
		ChannelCallback(Channel<T> &c, const T &v);
		void run();
};

/**
 * This Callback wakes a Microthread using a Suspender when run. The
 * Suspender's wakeup method is invoked with the value given at the
 * constructor.
 */
template<class T> class SuspendCallback : public Callback {
	private:
		Reference<Suspender<T> > suspender;
		T value;
	public:
		/**
		 * SuspendCallback constructor.
		 * @param s is the Suspender to be used
		 * @param v is the object to be passed to Suspender.wakeup()
		 * @see Suspender
		 */
		SuspendCallback(Suspender<T> &s, const T &v);
		void run();
		/**
		 * @return true if the Suspender is already wakened
		 */
		bool iscompleted();
};

/**
 * This Callback may be used to invoke multiple callbacks instead of one.
 */
class CallbackChain : public Callback {
	private:
		std::list<Callback*> callbacks;
	public:
		/**
		 * Constructs an empty CallbackChain.
		 */
		CallbackChain();
		/**
		 * Constructs a CallbackChain containing the single Callback cb.
		 * Make sure that cb was allocated using new as it will be
		 * destroyed by the destructor of the CallbackChain using
		 * delete.
		 * @param cb is put in the CallbackChain
		 */
		CallbackChain(Callback *cb);
		/**
		 * Adds a Callback to the CallbackChain. Make sure that cb was
		 * allocated using new as it will be destroyed by the destructor
		 * of the CallbackChain using delete.
		 * @param cb is put in the CallbackChain
		 */
		void add(Callback *cb);
		void run();
		~CallbackChain();
};

/**
 * This Callback just throws an exception when invoked.
 */
template<class T> class ExceptionCallback : public Callback {
	private:
		T exception;
	public:
		/**
		 * Constructs an ExceptionCallback that throws ex when invoked.
		 * @param ex is the exception to throw
		 */
		ExceptionCallback(T ex);
		void run();
};

#endif
