/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef __WIN32__
/* repair min and max */
#include "win32.h"
#endif
#include <algorithm>
#include "channel.h"
#include "currentthread.h"

bool BaseChannel::MSGIDComparator::operator()(const BaseChannel &a,
		const BaseChannel &b) const {
	return a.messageid < b.messageid;
}

void ChannelManager::notify() {
	/* datalock is acquired. */
	assert(this->datalock.tryacquire() == false);
	if(this->waiter) {
		Microthread &m(getcurrentthread());
		m.scheduleme();
		/* datalock is expected to be acquired and will be released. */
		m.swapto(*this->waiter);
	} else
		this->datalock.release();
}

void ChannelGroup::addChannel(BaseChannel &chan) {
	this->channels.insert(&chan);
}

void ChannelGroup::removeChannel(BaseChannel &chan) {
	this->channels.erase(&chan);
}

template<class F> class depointer_binary_function
		: public std::binary_function<typename F::first_argument_type*,
				typename F::second_argument_type*,
				typename F::result_type> {
	private:
		typedef typename depointer_binary_function::result_type
				result_type;
		typedef typename depointer_binary_function::first_argument_type
				first_argument_type;
		typedef typename depointer_binary_function::second_argument_type
				second_argument_type;
		F fun;
	public:
		explicit depointer_binary_function(F f) : fun(f) {}
		result_type operator() (first_argument_type a,
				second_argument_type b) const {
			assert(a != NULL);
			assert(b != NULL);
			return this->fun(*a, *b);
		}
};

BaseChannel *ChannelGroup::unlockedtryselect() const {
	assert(this->manager.datalock.tryacquire() == false);
	BaseChannel *const result(*min_element(this->channels.begin(),
			this->channels.end(),
			depointer_binary_function<BaseChannel::MSGIDComparator>(
					BaseChannel::MSGIDComparator())));
	return result->nextid() == std::numeric_limits<MessageID>::max()
			? NULL : result;
}

BaseChannel *ChannelGroup::tryselect() const {
	WITHACQUIRED(this->manager.datalock);
	return this->unlockedtryselect();
}

BaseChannel &ChannelGroup::select() const {
	WITHACQUIRED(this->manager.datalock);
	{
		BaseChannel *const result(this->unlockedtryselect());
		if(result)
			return *result;
	}
	assert(this->manager.waiter == NULL);
	this->manager.waiter = &getcurrentthread();
	for(;;) {
		this->manager.datalock.release();
		this->manager.waiter->delayme();
		/* The Lock will be acuired at this point. */
		assert(this->manager.datalock.tryacquire() == false);
		{
			BaseChannel *const result(this->unlockedtryselect());
			if(result) {
				this->manager.waiter = NULL;
				return *result;
			}
		}
	}
}
