/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_CHANNEL_H
#define LIBMUTH_CHANNEL_H

#ifdef __WIN32__
/* repair min and max */
#include "win32.h"
#endif
#include <limits>
#include <memory>
#include <assert.h>
#include "channel_decls.h"
#include "microthread.h"
#include "currentthread.h"

inline MessageID ChannelManager::genid() {
	assert(this->datalock.tryacquire() == false);
	return this->lastid++;
}

inline ChannelManager::ChannelManager() : lastid(0), waiter(NULL) {
}

inline BaseChannel::BaseChannel(ChannelManager &m)
		: manager(m), messageid(std::numeric_limits<MessageID>::max()) {
}

inline MessageID BaseChannel::nextid() const {
	return this->messageid;
}

template<class T> inline Channel<T>::Channel(ChannelManager &m)
		: BaseChannel(m) {
}

template<class T> void Channel<T>::send(T data) {
	this->manager.datalock.acquire();
	const MessageID msgid(this->manager.genid());
	this->queue.push(new std::pair<MessageID,T>(msgid, data));
	if(this->messageid == std::numeric_limits<MessageID>::max()) {
		this->messageid = msgid;
		/* Expects the lock to be acquired and releases it. */
		this->manager.notify();
	} else
		this->manager.datalock.release();
}

template<class T> T Channel<T>::receive() {
	WITHACQUIRED(this->manager.datalock);
	if(this->messageid == std::numeric_limits<MessageID>::max()) {
		assert(this->manager.waiter == NULL);
		this->manager.waiter = &getcurrentthread();
		this->manager.datalock.release();
		this->manager.waiter->delayme();
		/* The lock will be reacquired here. */
		assert(this->manager.datalock.tryacquire() == false);
		this->manager.waiter = NULL;
	}
	assert(this->messageid != std::numeric_limits<MessageID>::max());
	assert(this->queue.empty() == false);
	/* Credits to ##C++. */
	const T result((std::auto_ptr<std::pair<MessageID,T> >
			(this->queue.front()))->second);
	this->queue.pop();
	if(this->queue.empty())
		this->messageid = std::numeric_limits<MessageID>::max();
	else
		this->messageid = this->queue.front()->first;
	return result;
}

template<class T> inline bool Channel<T>::isempty() const {
	return this->messageid == std::numeric_limits<MessageID>::max();
}

inline ChannelGroup::ChannelGroup(ChannelManager &m) : manager(m) {
}

#endif
