/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_CHANNEL_DECLS_H
#define LIBMUTH_CHANNEL_DECLS_H

#include <list>
#include <set>
#include <utility>
#include <queue>
#ifndef __WIN32__
#include <stdint.h>
#else
#include "win32.h"
#endif
#include "lock.h"

typedef uint64_t MessageID;

class BaseChannel;
class Microthread;

/**
 * Each microthread that wants to receive messages needs exactly one
 * ChannelManager. It maintains a queue of all pending messages to be received.
 * All channels on a manager must be destroyed before destroying the manager.
 */
class ChannelManager {
	private:
		Lock datalock;
		MessageID lastid;
		Microthread *waiter;
		/**
		 * Creates messageids. Lock yourself.
		 * @returns the next unique monotonically increasing message id
		 */
		MessageID genid();
		/**
		 * Tells a possibly waiting ChannelGroup about arrived messages.
		 * datalock must be acquired while calling the method and will
		 * be released.
		 */
		void notify();
	public:
		/**
		 * ChannelManager constructor. Each channel that wants to use
		 * channels has to have exactly one ChannelManager. Just create
		 * it and pass it to channels and channelgroups.
		 */
		ChannelManager();
		friend class BaseChannel;
		friend class ChannelGroup;
		template<class T> friend class Channel;
};

/**
 * Common baseclass for all templated channels. Not instanciatable. Internal
 * use only.
 */
class BaseChannel {
	protected:
		ChannelManager &manager;
		MessageID messageid;
		/**
		 * BaseChannel constructor.
		 * @param m is the manager to register with
		 */
		BaseChannel(ChannelManager &m);
		/**
		 * Finds the next message.
		 * @returns the id of the next message if there is one else the
		 *          largest possible id.
		 */
		MessageID nextid() const;
		/**
		 * Binary functor for comparing channels. Channels with no
		 * messages are considered larger than channels with messages.
		 * If both channels have messages to be delivered the channel
		 * that received the message earlier is smaller.
		 */
		struct MSGIDComparator
				: public std::binary_function<BaseChannel,
						BaseChannel, bool> {
			/**
			 * Comparison operator.
			 */
			bool operator()(const BaseChannel& a,
				const BaseChannel &b) const;
		};
	public:
		friend class ChannelGroup;
};

/**
 * Channels can be used for micorthreads to communicate. They must be registered
 * with a ChannelManager.
 */
template<class T> class Channel : public BaseChannel {
	private:
		std::queue<std::pair<MessageID, T>*> queue;
	public:
		/**
		 * Channel constructor.
		 * @param m is the ChannelManager to register with.
		 */
		Channel(ChannelManager &m);
		/**
		 * Sends a message to this channel. Any microthread may invoke
		 * this method at any time.
		 * @param data is the message to be sent
		 */
		void send(T data);
		/**
		 * Receives the next (chronological) message from this channel.
		 * Only the microthread owning the corresponding manager may
		 * invoke this method.
		 * @returns the message
		 */
		T receive();
		/**
		 * Checks whether there are messages waiting to be delivered.
		 * This operation is not locked as it is used internally.
		 * @returns true if there are no messages waiting
		 */
		bool isempty() const;
};

/**
 * ChannelGroups are used to receive messages from multiple channels
 * simultaneously. Channels can be added and removed to and from a group using
 * addChannel() and removeChannel(). The methods select() and tryselect() can
 * be used to find a channel with pending messages. A group may only be used by
 * exactly one microthread.
 */
class ChannelGroup {
	private:
		ChannelManager &manager;
		std::set<BaseChannel*> channels;
		BaseChannel *unlockedtryselect() const;
	public:
		/**
		 * ChannelGroup constructor.
		 * @param m is the manager to register with.
		 */
		ChannelGroup(ChannelManager &m);
		/**
		 * Adds given channel to this group. A tryselect() or select()
		 * will then take this channel into account.
		 * @param chan is the channel to add
		 */
		void addChannel(BaseChannel &chan);
		/**
		 * Removes given channel from this group. A tryselect() or
		 * select() will no longer respond to messages on this channel.
		 * @param chan is the channel to remove
		 */
		void removeChannel(BaseChannel &chan);
		/**
		 * Tries to find a channel where messages are available. It is
		 * guarantueed that two messages arriving on one channel are
		 * selected chronologically. It is furthermore guaranteed that
		 * two messages sent on two channels within this group are
		 * selected chronologically. This only takes message arrival
		 * into account. If two microthreads try to concurrently send
		 * messages to channels the one wining the lock of the manager
		 * will be first.
		 * @returns a pointer to the selected channel if a message is
		 *          available and NULL otherwise
		 */
		BaseChannel *tryselect() const;
		/**
		 * Blockingly finds a channel where messages are available.
		 * Please read the docs about tryselect(). If there are no
		 * messages available the current microthread is suspended
		 * until messages are available.
		 * @returns a reference to the selected channel
		 */
		BaseChannel &select() const;
};

#endif
