/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cstdlib>
#include <assert.h>
#ifndef __WIN32__
#include <ucontext.h>
#endif
#include <stdlib.h>
#ifdef USE_VALGRIND
extern "C" {
#include <valgrind.h>
};
#include <stdint.h>
#endif
#include "coroutine.h"
#include "threadlocal.h"
#include "refcounter.h"

static ThreadLocal<std::pair<NewCoroutine*, ucontext_t*> > &get_booter() {
	static ThreadLocal<std::pair<NewCoroutine*, ucontext_t*> > booter;
	return booter;
}

inline int Coroutine::swapcontext(Coroutine &o) {
	assert(this->returned == false);
	assert(this->from == NULL);
	assert(o.returned == false);
	assert(o.from == this);
	return ::swapcontext(&this->context, &o.context);
}

inline int Coroutine::setcontext() {
	assert(this->returned == false);
	assert(this->from != NULL);
	return ::setcontext(&this->context);
}

void Coroutine::cleanupfrom() {
	assert(this->returned == false);
	if(this->from) {
		if(this->from->returned)
			this->from->destroyme();
		else
			this->from->running.release();
		this->from = NULL;
	}
}

void Coroutine::destroyme() {
	assert(0);
}

Coroutine::Coroutine() : from(NULL), returned(false) {
	this->context.uc_stack.ss_sp = NULL;
	if(::getcontext(&this->context) < 0)
		assert(0);
	this->running.acquire();
}

Coroutine::~Coroutine() {
}

void Coroutine::swapto(Coroutine &o, bool acquired) {
	assert(this->from == NULL);
	assert(this->returned == false);
	if(!acquired)
		o.running.acquire();
	assert(o.from == NULL);
	assert(o.returned == false);
	o.from = this;
	if(this->swapcontext(o) < 0)
		assert(0);
	this->cleanupfrom();
}

void NewCoroutine::destroyme() {
	this->decref();
}

void NewCoroutine::boot() {
	NewCoroutine *t;
	{
		std::pair<NewCoroutine*,ucontext_t*> &p(*get_booter());
		t = p.first;
		assert(t != NULL);
		assert(p.second != NULL);
		/* This will resume in NewCoroutine::NewCoroutine(). */
		::swapcontext(&t->context, p.second);
	}
	t->cleanupfrom();
	t->start();
}

NewCoroutine::NewCoroutine(int stacksize) {
	this->incref();
	this->context.uc_stack.ss_sp = new char[stacksize];
	this->context.uc_stack.ss_size = stacksize;
	this->running.release();
#ifdef USE_VALGRIND
	this->valgrind_stackid = VALGRIND_STACK_REGISTER(
			(uint64_t)this->context.uc_stack.ss_sp,
			(uint64_t)this->context.uc_stack.ss_sp +
				this->context.uc_stack.ss_size);
#endif
	::makecontext(&this->context, (void(*)())NewCoroutine::boot, 0);
	ucontext_t ctx[1];
	::getcontext(ctx);
	std::pair<NewCoroutine*, ucontext_t*> p(this, ctx);
	get_booter() = &p;
	/* This will continue with NewCoroutine::boot(). It will immediately
	 * swap back after reading the pair. */
	::swapcontext(ctx, &this->context);
}

NewCoroutine::~NewCoroutine() {
	assert(this->returned == true);
	assert(this->from == NULL);
	this->running.release();
#ifdef USE_VALGRIND
	VALGRIND_STACK_DEREGISTER(this->valgrind_stackid);
#endif
	delete[] (char*)this->context.uc_stack.ss_sp;
}

void NewCoroutine::returnto(Coroutine &o, bool acquired) {
	assert(this->from == NULL);
	assert(this->returned == false);
	if(!acquired)
		o.running.acquire();
	assert(o.from == NULL);
	assert(o.returned == false);
	this->returned = true;
	o.from = this;
	o.setcontext();
	assert(0);
}

