/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_COROUTINE_DECLS_H
#define LIBMUTH_COROUTINE_DECLS_H

#ifndef __WIN32__
#include <ucontext.h>
#else
#include "win32.h"
#endif
#include "lock.h"
#include "refcounter_decls.h"

class NewCoroutine;

/**
 * Internal coroutine interface class.
 */
class Coroutine {
	private:
		Lock running;
		Coroutine *from;
		bool returned;
		int swapcontext(Coroutine&);
		int setcontext();
		void cleanupfrom();
	protected:
		/**
		 * Internal context datastructure. It is not private because
		 * it is accessed in NewCoroutine.
		 */
		ucontext_t context;
		/**
		 * Internal function return interface. Do not use.
		 */
		virtual void destroyme();
	public:
		Coroutine();
		virtual ~Coroutine();
		/**
		 * Acquires this coroutine before a swapto() or returnto() when
		 * other datastructures need to be modified. The acquired
		 * parameter of the next swapto() or returnto() must be true.
		 */
		void acquire();
		/**
		 * Swaps the context from this coroutine to the given coroutine.
		 * @param o is the coroutine being swapped to
		 * @param acquired must be true if this object was previously
		 *        acquired using acquire()
		 */
		void swapto(Coroutine &o, bool acquired=false);
		friend class NewCoroutine;
};

/**
 * Internal coroutine class for coroutines that can be created without a
 * corresponding thread.
 */
class NewCoroutine : public Coroutine, public RefCounter {
	private:
		virtual void destroyme();
		static void boot();
#ifdef USE_VALGRIND
		int valgrind_stackid;
#endif
	public:
		/**
		 * NewCoroutine constructor.
		 * @param stacksize is the size of the stack to be allocated
		 *        for this coroutine
		 */
		NewCoroutine(int stacksize=4096*1024);
		~NewCoroutine();
		/**
		 * Lets this coroutine return to the given coroutine.
		 * @param o is the coroutine to be continued
		 * @param acquired must be true if this object was previously
		 *        acquired using acquire()
		 * @returns never
		 */
		void returnto(Coroutine &o, bool acquired=false);
		/**
		 * This is the coroutines main function. It has to be overridden
		 * by subclasses.
		 */
		virtual void start()=0;
};

#endif
