/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef USE_EPOLL
#error this file must be compiled with -DUSE_EPOLL
#endif

#include <map>
#include <queue>
#include <limits>
#include <assert.h>
#include <errno.h>
#include <sys/epoll.h>
#include <unistd.h>
#include "epollscheduler.h"
#include "globalqueue.h"
#include "callback_decls.h"
#include "timeval.h"
#include "lock.h"

inline void EpollScheduler::epoll_ctl(int op, int fd,
		struct epoll_event *event) {
	const int r(::epoll_ctl(this->efd, op, fd, event));
	assert(r >= 0);
}

EpollScheduler::EpollScheduler(BlockingQueue<Microthread*> *q)
		: BaseScheduler(q ? q : &getglobalqueue()) {
	{
		int p[2];
		if(pipe(p) < 0)
			assert(0);
		this->rpipe = p[0];
		this->wpipe = p[1];
	}
	this->efd = epoll_create(32);
	if(this->efd < 0)
		assert(0);
	struct epoll_event e;
	e.events = EPOLLIN;
	e.data.fd = this->rpipe;
	this->epoll_ctl(EPOLL_CTL_ADD, this->rpipe, &e);
}

EpollScheduler &EpollScheduler::create(BlockingQueue<Microthread*> *q) {
	return *new EpollScheduler(q);
}

void EpollScheduler::notifyme() {
	if(this->notifylock.tryacquire()) {
		const int n(::write(this->wpipe, "\0", 1));
		assert(n == 1);
	}
}


void EpollScheduler::addReadCallback(Callback &cb, int fd) {
	assert(fd >= 0);
	WITHACQUIRED(this->datalock);
	if(this->rfds[fd] != NULL) {
		assert(this->rfds[fd]->iscompleted());
		delete this->rfds[fd];
	} else {
		struct epoll_event e;
		e.data.fd = fd;
		if(this->wfds[fd] == NULL) {
			e.events = EPOLLIN;
			this->epoll_ctl(EPOLL_CTL_ADD, fd, &e);
		} else {
			e.events = EPOLLIN|EPOLLOUT;
			this->epoll_ctl(EPOLL_CTL_MOD, fd, &e);
		}
	}
	this->rfds[fd] = &cb;
}

void EpollScheduler::addWriteCallback(Callback &cb, int fd) {
	assert(fd >= 0);
	WITHACQUIRED(this->datalock);
	if(this->wfds[fd] != NULL) {
		assert(this->wfds[fd]->iscompleted());
		delete this->wfds[fd];
	} else {
		struct epoll_event e;
		e.data.fd = fd;
		if(this->rfds[fd] == NULL) {
			e.events = EPOLLOUT;
			this->epoll_ctl(EPOLL_CTL_ADD, fd, &e);
		} else {
			e.events = EPOLLIN|EPOLLOUT;
			this->epoll_ctl(EPOLL_CTL_MOD, fd, &e);
		}
	}
	this->wfds[fd] = &cb;
}

void EpollScheduler::addWaitCallback(Callback &cb, const timeval &tv) {
	bool notify(false);
	{
		WITHACQUIRED(this->datalock);
		if(this->waits.isempty()) {
			notify = true;
			this->waits.put(cb, tv);
		} else {
			const Timeval now(TimevalNow);
			const Timeval old(this->waits.delay(now));
			this->waits.put(cb, tv);
			if(old != this->waits.delay())
				notify = true;
		}
	}
	if(notify)
		this->notifyme();
}

void EpollScheduler::run() {
	int n;
	std::map<int, Callback*>::iterator j;
	std::queue<Callback*> callbackqueue;
	for(;;) {
		do {
			errno = 0;
			this->datalock.acquire();
			if(this->waits.cleanup()) {
				static const Timeval maxmilliseconds(
						std::numeric_limits<time_t>::
							max()/1000);
				const Timeval t(this->waits.delay());
				if(t < maxmilliseconds)
					n = t.milliseconds();
				else
					n = (std::numeric_limits<time_t>::
							max() / 1000) * 1000;
			} else
				n = -1;
			this->datalock.release();
			n = ::epoll_wait(this->efd, this->events,
					EPOLL_MAX_EVENTS, n);
		} while(n < 0 && errno == EINTR);

		/*
		 * No matter whether we succeed in getting the lock it will now
		 * be acquired, so no other microthread will acquire it and send
		 * stuff to the pipe. This will save us some iterations of this
		 * loop.
		 */
		this->notifylock.tryacquire();
		this->datalock.acquire();
		assert(n >= 0);
		assert(n <= EPOLL_MAX_EVENTS);
		for(int i(0);i<n;++i) {
			if((this->events[i].events & (EPOLLIN|EPOLLOUT))
					== (EPOLLIN|EPOLLOUT)) {
				j = this->rfds.find(this->events[i].data.fd);
				assert(j != this->rfds.end());
				assert(j->second != NULL);
				callbackqueue.push(j->second);
				this->rfds.erase(j);
				j = this->wfds.find(this->events[i].data.fd);
				assert(j != this->wfds.end());
				assert(j->second != NULL);
				callbackqueue.push(j->second);
				this->wfds.erase(j);
				this->epoll_ctl(EPOLL_CTL_DEL,
						this->events[i].data.fd,
						this->events+i);
			} else if(this->events[i].events & EPOLLIN) {
				if(this->events[i].data.fd == this->rpipe) {
					/* This is a writeonly buffer. */
					char devnull[1];
					const int ret(::read(this->rpipe,
							devnull, 1));
					assert(ret == 1);
				} else {
					j = this->rfds.find(this->events[i].
							data.fd);
					assert(j != this->rfds.end());
					assert(j->second != NULL);
					callbackqueue.push(j->second);
					this->rfds.erase(j);
					if(this->wfds[this->events[i].data.fd]
							== NULL)
						this->epoll_ctl(EPOLL_CTL_DEL,
								this->events[i]
								.data
								.fd,
								this->events+i);
					else {
						this->events[i].data.fd =
							EPOLLOUT;
						this->epoll_ctl(EPOLL_CTL_MOD,
								this->events[i]
								.data
								.fd,
								this->events+i);
					}
				}
			} else if(this->events[i].events & EPOLLOUT) {
				j = this->wfds.find(this->events[i].data.fd);
				assert(j != this->wfds.end());
				assert(j->second != NULL);
				callbackqueue.push(j->second);
				this->wfds.erase(j);
				if(this->rfds[this->events[i].data.fd] == NULL)
					this->epoll_ctl(EPOLL_CTL_DEL,
							this->events[i].data.fd,
							this->events+i);
				else {
					this->events[i].data.fd = EPOLLIN;
					this->epoll_ctl(EPOLL_CTL_MOD,
							this->events[i].data.fd,
							this->events+i);
				}
			}
		}
		if(!this->waits.isempty()) {
			Timeval tv(TimevalNow);
			while(Callback *cb = this->waits.get(tv))
				callbackqueue.push(cb);
		}
		this->datalock.release();
		/* Empty the callbackqueue here. */
		while(!callbackqueue.empty()) {
			Callback *cb(callbackqueue.front());
			callbackqueue.pop();
			if(!cb->iscompleted())
				cb->run();
			delete cb;
		}
		while(!this->queue.isempty()) {
			this->scheduleme();
			try {
				this->delayme();
			} catch(...) {
				/* See comment below. */
				this->notifylock.release();
				throw;
			}
		}
		/*
		 * The lock was either acquired above or by another thread
		 * notifying us. So we release it to get notifies again.
		 */
		this->notifylock.release();
	}
}
