/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_EPOLLSCHEDULER_H
#define LIBMUTH_EPOLLSCHEDULER_H

#ifndef USE_EPOLL
#error this file must be compiled with -DUSE_EPOLL
#endif

#include <map>
#include <sys/epoll.h>
#include "basescheduler_decls.h"
#include "callback_decls.h"
#include "lock.h"
#include "timeval_decls.h"
#include "waitqueue.h"

#define EPOLL_MAX_EVENTS 256

/**
 * Scheduler implementation using level triggered epoll.
 */
class EpollScheduler : public BaseScheduler {
	private:
		/*
		 * This lock locks all acceess to rfds, wfds and waits.
		 */
		Lock datalock;
		/*
		 * Whenever a Microthread wants to read or write from or to a
		 * filedescriptor it runs the scheduleReader or scheduleWriter
		 * method which will add the filedescriptor as key and the
		 * instance pointer as value to rfdsched or wfdsched. The pair
		 * will be discarded from the map when the filedescriptor
		 * becomes readable or writeable.
		 */
		std::map<int, Callback*> rfds, wfds;
		WaitQueue waits;

		Lock notifylock;
		int rpipe, wpipe;
		void notifyme();
		/**
		 * epoll filedescriptor created by epoll_create().
		 */
		int efd;
		void epoll_ctl(int op, int fd, struct epoll_event *event);
		struct epoll_event events[EPOLL_MAX_EVENTS];
	protected:
		/**
		 * Constructs a EpollScheduler running on queue q.
		 * @param q is the queue the scheduler needs to insert itself
		 */
		EpollScheduler(BlockingQueue<Microthread*> *q=NULL);
	public:
		/**
		 * Named constructor for EpollScheduler. The object is
		 * automatically deleted when run() returns.
		 * @param q is the queue the scheduler needs to insert itself
		 */
		static EpollScheduler &create(
				BlockingQueue<Microthread*> *q=NULL);
		void addReadCallback(Callback &cb, int fd);
		void addWriteCallback(Callback &cb, int fd);
		void addWaitCallback(Callback &cb, const timeval &tv);
		void run();
};

#endif
