/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_IOSTREAM_H
#define LIBMUTH_IOSTREAM_H

#include <assert.h>
#include "iostream_decls.h"

template<class C, class T>
		BasicScheduledStreambuf<C, T>::BasicScheduledStreambuf(int f,
		BaseScheduler &s, std::ios_base::openmode m,
		std::streamsize insize, std::streamsize outsize) : fd(f),
		sched(s), mode(m) {
	// Do not allow opening a stream neither for reading nor for writing.
	assert(this->mode & (std::ios_base::in|std::ios_base::out));
	this->ibufsize = insize;
	if(this->mode & std::ios_base::in)
		this->ibuf = new char_type[insize];
	else
		this->ibuf = NULL;
	this->setg(this->ibuf, this->ibuf, this->ibuf);
	if(this->mode & std::ios_base::out) {
		this->obuf = new char_type[outsize];
		this->setp(this->obuf, this->obuf + outsize);
	} else {
		this->obuf = NULL;
		this->setp(NULL, NULL);
	}
}

template<class C, class T>
		BasicScheduledStreambuf<C, T>::~BasicScheduledStreambuf() {
	if(this->mode & std::ios_base::in)
		delete[] this->ibuf;
	if(this->mode & std::ios_base::out) {
		this->sync();
		delete[] this->obuf;
	}
}

template<class C, class T> typename BasicScheduledStreambuf<C, T>::int_type
		BasicScheduledStreambuf<C, T>::underflow() {
	assert(this->mode & std::ios_base::in);
	if(this->gptr() < this->egptr())
		return traits_type::to_int_type(*this->gptr());
	assert(this->ibufsize > 0);
	const int r(this->sched.read(this->fd, this->ibuf,
			this->ibufsize * sizeof(char_type)));
	if(r <= 0)
		return traits_type::eof();
	/* FIXME: this will result in problems showed by assertions: */
	assert(r % sizeof(char_type) == 0);
	this->setg(this->ibuf, this->ibuf, static_cast<char_type*>(
			static_cast<char*>(this->ibuf) + r));
	return traits_type::to_int_type(*this->gptr());
}

template<class C, class T> typename BasicScheduledStreambuf<C, T>::int_type
		BasicScheduledStreambuf<C, T>::pbackfail(int_type ch) {
	assert(this->mode & std::ios_base::in);
	if(this->gptr() != this->eback()) {
		this->gbump(-1);
		if(!traits_type::eq_int_type(ch, traits_type::eof()))
			*this->gptr() = traits_type::not_eof(ch);
		return traits_type::not_eof(ch);
	}
	return traits_type::eof();
}

template<class C, class T> int BasicScheduledStreambuf<C, T>::sync() {
	assert(this->mode & std::ios_base::out);
	const int bump(static_cast<int>(this->pptr() - this->pbase()));
	char *data(static_cast<char*>(this->pbase()));
	size_t n(static_cast<size_t>(static_cast<char*>(this->pptr()) - data));
	if(n == 0)
		return 0;
	int r(this->sched.write(this->fd, data, n));
	if(r <= 0)
		return -1;
	assert(n >= (size_t)r);
	while((size_t)r != n) {
		r = this->sched.write(this->fd, data, n);
		if(r <= 0)
			return -1;
		assert(n < (size_t)r);
		data += r;
		n -= r;
	}
	this->pbump(-bump);
	return 0;
}

template<class C, class T> typename BasicScheduledStreambuf<C, T>::int_type
		BasicScheduledStreambuf<C, T>::overflow(int_type ch) {
	assert(this->mode & std::ios_base::out);
	/* FIXME: maybe not really sync but write parts of the buffer. */
	if(this->sync() < 0)
		return traits_type::eof();
	if(traits_type::eq_int_type(ch, traits_type::eof()))
		/* FIXME: find out why we return not_eof(eof()). */
		return traits_type::not_eof(ch);
	return this->sputc(traits_type::to_char_type(ch));
}

#if 0
template<class C, class T> std::streamsize
		BasicScheduledStreambuf<C, T>::xsputn(const char_type *str,
		std::streamsize n) {
	assert(this->mode & std::ios_base::out);
	if(this->sync() < 0)
		return 0;
	const int r(this->sched.write(this->fd, str, n * sizeof(char_type)));
	if(r < 0)
		return 0;
	return r / sizeof(char_type);
}
#endif

template<class C, class T> inline BasicScheduledStreambuf<C, T>::operator int()
		const {
	return this->fd;
}

template<class C, class T> BasicScheduledIStream<C, T>::BasicScheduledIStream(
		int f, BaseScheduler &s, std::streamsize size)
		: std::basic_istream<C, T>(NULL),
		streambuf(f, s, std::ios_base::in, size) {
	this->init(&this->streambuf);
}

template<class C, class T> inline BasicScheduledIStream<C, T>::operator int()
		const {
	return (int)this->streambuf;
}

template<class C, class T> BasicScheduledOStream<C, T>::BasicScheduledOStream(
		int f, BaseScheduler &s, std::streamsize size)
		: std::basic_ostream<C, T>(NULL),
		streambuf(f, s, std::ios_base::out, 0, size) {
	this->init(&this->streambuf);
}

template<class C, class T> inline BasicScheduledOStream<C, T>::operator int()
		const {
	return (int)this->streambuf;
}

template<class C, class T> BasicScheduledIOStream<C, T>::BasicScheduledIOStream(
		int f, BaseScheduler &s, std::streamsize insize,
		std::streamsize outsize) : std::basic_iostream<C, T>(NULL),
		streambuf(f, s, std::ios_base::in | std::ios_base::out, insize,
		outsize) {
	this->init(&this->streambuf);
}

template<class C, class T> inline BasicScheduledIOStream<C, T>::operator int()
		const {
	return (int)this->streambuf;
}

#endif
