/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_IOSTREAM_DECLS_H
#define LIBMUTH_IOSTREAM_DECLS_H

#include <iostream>
#include "basescheduler_decls.h"

/**
 * A streambuf implementation working with this library. It takes the interface
 * of std::basic_streambuf and implements all the necessary methods to do
 * non-blocking IO in a way that seems to be blocking.
 */
template<class C=char, class T=std::char_traits<C> >
		class BasicScheduledStreambuf
		: public std::basic_streambuf<C, T> {
	private:
		typedef C char_type;
		typedef T traits_type;
		typedef typename traits_type::int_type int_type;
		int fd;
		BaseScheduler &sched;
		std::ios_base::openmode mode;
		std::streamsize ibufsize;
		char_type *ibuf;
		char_type *obuf;
	protected:
		/* istreambuf */
		/**
		 * std::basic_streambuf::underflow implementation.
		 * @returns -1 on error, 0 otherwise
		 */
		int_type underflow();
		/**
		 * std::basic_streambuf::pbackfail implementation.
		 * @param ch is the character to be prepended
		 * @returns eof() on failure not_eof(ch) otherwise
		 */
		int_type pbackfail(int_type ch);
		/* ostreambuf */
		/**
		 * std::basic_streambuf::sync implementation.
		 * @returns -1 on error, 0 otherwise
		 */
		int sync();
		/**
		 * std::basic_streambuf::overflow implementation.
		 * @param ch is the character to be appended
		 * @returns eof() on failure not_eof(ch) otherwise
		 */
		int_type overflow(int_type ch=traits_type::eof());
#if 0
		std::streamsize xsputn(const char_type *str, std::streamsize n);
#endif
	public:
		/**
		 * Constructor.
		 * @param f is the file descriptor to do IO on
		 * @param s is the scheduler to work with
		 * @param m is the open mode. @see std::ios_base
		 * @param insize is the buffer size for reading
		 * @param outsize is the buffer size for writing
		 */
		BasicScheduledStreambuf(int f, BaseScheduler &s,
				std::ios_base::openmode m = std::ios_base::in |
				std::ios_base::out, std::streamsize insize=4096,
				std::streamsize outsize=4096);
		/* TODO: avoid copying */
		~BasicScheduledStreambuf();
		/**
		 * Cast operator to get the file descriptor.
		 * @returns the file descriptor
		 */
		operator int() const;
};

/**
 * An istream implementation working with this library. It should be usable
 * like a normal STL istream. Blocking operations are wrapped to happen
 * non-blocking.
 */
template<class C=char, class T=std::char_traits<C> > class BasicScheduledIStream
		: public std::basic_istream<C, T> {
	private:
		BasicScheduledStreambuf<C, T> streambuf;
	public:
		/**
		 * Constructor.
		 * @param f is the file descriptor to do IO on
		 * @param s is the scheduler to work with
		 * @param size is the buffer size for reading
		 */
		BasicScheduledIStream(int f, BaseScheduler &s,
				std::streamsize size=4096);
		/**
		 * Cast operator to get the file descriptor.
		 * @returns the file descriptor
		 */
		operator int() const;
};

/**
 * An ostream implementation working with this library. It should be usable
 * like a normal STL ostream. Blocking operations are wrapped to happen
 * non-blocking.
 */
template<class C=char, class T=std::char_traits<C> > class BasicScheduledOStream
		: public std::basic_ostream<C, T> {
	private:
		BasicScheduledStreambuf<C, T> streambuf;
	public:
		/**
		 * Constructor.
		 * @param f is the file descriptor to do IO on
		 * @param s is the scheduler to work with
		 * @param size is the buffer size for writing
		 */
		BasicScheduledOStream(int f, BaseScheduler &s,
				std::streamsize size=4096);
		/**
		 * Cast operator to get the file descriptor.
		 * @returns the file descriptor
		 */
		operator int() const;
};

/**
 * An iostream implementation working with this library. It should be usable
 * like a normal STL iostream. Blocking operations are wrapped to happen
 * non-blocking.
 */
template<class C=char, class T=std::char_traits<C> >
		class BasicScheduledIOStream
		: public std::basic_iostream<C, T> {
	private:
		BasicScheduledStreambuf<C, T> streambuf;
	public:
		/**
		 * Constructor.
		 * @param f is the file descriptor to do IO on
		 * @param s is the scheduler to work with
		 * @param insize is the buffer size for reading
		 * @param outsize is the buffer size for writing
		 */
		BasicScheduledIOStream(int f, BaseScheduler &s,
				std::streamsize insize=4096,
				std::streamsize outsize=4096);
		/**
		 * Cast operator to get the file descriptor.
		 * @returns the file descriptor
		 */
		operator int() const;
};

typedef BasicScheduledStreambuf<> ScheduledStreambuf;
typedef BasicScheduledIStream<> ScheduledIStream;
typedef BasicScheduledOStream<> ScheduledOStream;
typedef BasicScheduledIOStream<> ScheduledIOStream;

#endif
