/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_LOCK_H
#define LIBMUTH_LOCK_H

#include "config.h"
#ifdef USE_PTHREAD
#include "pthreadlock.h"
typedef PthreadLock Lock;
#else
#include "dummylock.h"
typedef DummyLock Lock;
#endif

/**
 * This class acquires a lock when instantiated and releases it when the
 * destructor is run. This can be used to implement locked sections without
 * catching every possible thing and rethrow it as the destructor is always
 * run, when the closure ends.
 */
class WithAcquired {
	private:
		Lock &lock;
	public:
		/**
		 * Constructs a WithAcquired object by acquiring the given lock.
		 * @param l is the lock to be acquired
		 */
		inline WithAcquired(Lock &l) : lock(l) { this->lock.acquire(); }
		/**
		 * The destructor releases the lock.
		 */
		inline ~WithAcquired() { this->lock.release(); }
};

#define WITHACQUIRED(lock) const WithAcquired withacquired(lock)

#endif
