/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_MICROTHREAD_H
#define LIBMUTH_MICROTHREAD_H

#include "blockingqueue.h"
#include "channel_decls.h"
#include "coroutine_decls.h"

class Runner;
class Callback;

/**
 * This class defines an interface and necessary methods for Microthreads.
 */
class Microthread : public NewCoroutine {
	private:
		Runner *current_runner;
		friend class Runner;

		Callback *callback;
		bool runcallback();
	protected:
		/**
		 * The queue to be used with scheduleme(). This is internal
		 * and should not be used. It is not private because it is
		 * used in scheduler implementations.
		 */
		BlockingQueue<Microthread*> &queue;
		/**
		 * Construct a Microthread running on queue q.
		 * @param q is the queue the Microthread needs to insert itself
		 */
		Microthread(BlockingQueue<Microthread*> *q=NULL);
		/**
		 * Boots the Microthread.
		 */
		void start();
	public:
		/**
		 * Manager object for channels associated with this
		 * microthread. It should be passed to the Channel constructor.
		 */
		ChannelManager channelmanager;
		/**
		 * This method is run when the Microthread gets started. It
		 * must be overridden by all derived classes.
		 */
		virtual void run()=0;
		virtual ~Microthread() {}
		/**
		 * Schedule this Microthread for continuation. If this
		 * Microthread currently runs it should immediately call
		 * .delayme().
		 */
		void scheduleme();
		/**
		 * Discontinues executing this Microthread for now. It may later
		 * be continued by scheduleme() or special swapto() calls.
		 * @throws ThreadExitException
		 */
		void delayme();
		using Coroutine::swapto;
		/**
		 * Discontinues executing this Microthread and continue
		 * execution of the Microthread o. Apart from directly passing
		 * control to o this works like delayme().
		 * @param o is the Microthread to be continued
		 * @param acquired the internal lock is not acquired of this is
		 * 		   set to true, ignore this if you don't know
		 * 		   about internal locks
		 * @throws ThreadExitException
		 */
		void swapto(Microthread &o, bool acquired=false);
		/* TODO: a better name is needed */
		/**
		 * Sets a Callback to be executed at the next time this
		 * Microthread continues execution. The Callback will be
		 * executed within this Microthread and thus may throw
		 * exceptions to terminate the Microthread.
		 * @param cb is the callback to be executed
		 */
		void setcallback(Callback *cb);
};

inline void Microthread::scheduleme() {
	this->queue.put(this);
}

#endif
