/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_PTHREADTHREADLOCAL_H
#define LIBMUTH_PTHREADTHREADLOCAL_H

#include <assert.h>
#include <pthread.h>

#ifndef USE_PTHREAD
#error this file must be compiled with -DUSE_PTHREAD
#endif

/**
 * Internal pthread threadlocl adaption class. This really contains a
 * pthread_key_t.
 */
template<class T> class PthreadThreadLocal {
	private:
		pthread_key_t key;
	public:
		/**
		 * ThreadLocal constructor.
		 */
		PthreadThreadLocal() {
				const int r(pthread_key_create(&this->key,
						NULL));
				assert(r == 0);
			}
		/**
		 * ThreadLocal destructor.
		 */
		~PthreadThreadLocal() {
				const int r(pthread_key_delete(this->key));
				assert(r == 0);
			}
		/**
		 * Assignment operator which makes this object look like a
		 * pointer.
		 * @param v is the pointer to be stored
		 * @returns *this
		 */
		PthreadThreadLocal &operator=(T *v) {
				const int r(pthread_setspecific(this->key,
						static_cast<void*>(v)));
				assert(r == 0);
				return *this;
			}
		/**
		 * Dereferencing operator which makes this object look like a
		 * pointer.
		 * @returns a reference to the stored pointer
		 */
		T &operator*() {
				return *(static_cast<T*>(pthread_getspecific(
							this->key)));
			}
		/**
		 * Constant dereferencing operator which makes this object look
		 * like a pointer.
		 * @returns a reference to the stored pointer
		 */
		T &operator*() const {
				return *(static_cast<T*>(pthread_getspecific(
							this->key)));
			}
};

#endif
