/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_REFCOUNTER_H
#define LIBMUTH_REFCOUNTER_H

#include <assert.h>
#include "refcounter_decls.h"

inline void RefCounter::addWeakRef(WeakRefBase &wr) {
	WITHACQUIRED(this->datalock);
	this->notifier.insert(&wr);
	this->cond.notify();
}

inline void RefCounter::removeWeakRef(WeakRefBase &wr) {
	WITHACQUIRED(this->datalock);
	this->notifier.erase(&wr);
	this->cond.notify();
}

inline void RefCounter::notifyall() {
	/* datalock is acquired from ~RefCounter() */
	while(!this->notifier.empty()) {
		WeakRefBase *const t(*this->notifier.begin());
		if(t->datalock.tryacquire()) {
			this->notifier.erase(this->notifier.begin());
			t->notify();
			t->datalock.release();
		} else
			this->cond.wait();
	}
}

inline RefCounter::~RefCounter() {
	/* datalock is acquired in decref. */
	this->notifyall();
	this->datalock.release();
}

inline RefCounter::RefCounter() : cond(datalock), refcount(0) {
}

inline void RefCounter::incref() {
	WITHACQUIRED(this->datalock);
	++this->refcount;
}

inline void RefCounter::decref() {
	this->datalock.acquire();
	if(!--this->refcount)
		/* The destructor will release the lock. */
		delete this;
	else
		this->datalock.release();
}

template<class RC> inline Reference<RC>::Reference(RC *obj) : object(obj) {
	assert(this->object != NULL);
	this->object->incref();
}

template<class RC> inline Reference<RC>::Reference(const Reference &other)
		: object(other.object) {
	this->object->incref();
}

template<class RC> inline Reference<RC> &Reference<RC>::operator=(RC *obj) {
	assert(obj != NULL);
	obj->incref();
	this->object->decref();
	this->object = obj;
	return *this;
}

template<class RC> inline Reference<RC> &Reference<RC>::operator=(const
		Reference &other) {
	other.object->incref();
	this->object->decref();
	this->object = other.object;
	return *this;
}

template<class RC> inline bool Reference<RC>::operator==(const Reference
		&other) const {
	return this->object == other.object;
}

template<class RC> inline bool Reference<RC>::operator!=(const Reference
		&other) const {
	return this->object != other.object;
}

template<class RC> inline RC &Reference<RC>::operator*() const {
	return *this->object;
}

template<class RC> inline RC *Reference<RC>::operator->() const {
	return this->object;
}

template<class RC> inline Reference<RC>::~Reference() {
	this->object->decref();
}

inline WeakRefBase::WeakRefBase() {
}

template<class RC> void WeakReference<RC>::notify() {
	/* locked in RefCounter::notifyall */
	this->object = NULL;
}

template<class RC> inline WeakReference<RC>::WeakReference() : object(NULL) {
}

template<class RC> inline WeakReference<RC>::WeakReference(const WeakReference
		&other) : object(other.object) {
	if(this->object)
		this->object->addWeakRef(*this);
}

template<class RC> inline WeakReference<RC>::WeakReference(const Reference<RC>
		&other) : object(other.object) {
	assert(this->object != NULL);
	this->object->addWeakRef(*this);
}

template<class RC> inline WeakReference<RC>::WeakReference(RC *obj)
		: object(obj) {
	if(this->object)
		this->object->addWeakRef(*this);
}

template<class RC> inline WeakReference<RC> &WeakReference<RC>::operator=(const
		WeakReference &other) {
	WITHACQUIRED(this->datalock);
	if(this->object != other.object) {
		if(this->object)
			this->object->removeWeakRef(*this);
		this->object = other.object;
		if(this->object)
			this->object->addWeakRef(*this);
	}
	return *this;
}

template<class RC> inline WeakReference<RC> &WeakReference<RC>::operator=(const
		Reference<RC> &other) {
	WITHACQUIRED(this->datalock);
	if(this->object != other.object) {
		if(this->object)
			this->object->removeWeakRef(*this);
		this->object = other.object;
		assert(this->object != NULL);
		this->object->addWeakRef(*this);
	}
	return *this;
}

template<class RC> inline bool WeakReference<RC>::operator==(const WeakReference
		&other) const {
	WITHACQUIRED(this->datalock);
	return this->object == other.object;
}

template<class RC> inline bool WeakReference<RC>::operator==(const Reference<RC>
		&other) const {
	WITHACQUIRED(this->datalock);
	return this->object == other.object;
}

template<class RC> inline bool WeakReference<RC>::operator!=(const WeakReference
		&other) const {
	WITHACQUIRED(this->datalock);
	return this->object != other.object;
}

template<class RC> inline bool WeakReference<RC>::operator!=(const Reference<RC>
		&other) const {
	WITHACQUIRED(this->datalock);
	return this->object != other.object;
}

template<class RC> inline Reference<RC> WeakReference<RC>::operator*() const {
	WITHACQUIRED(this->datalock);
	if(!this->object)
		throw NullPointerException();
	return Reference<RC>(this->object);
}

template<class RC> WeakReference<RC>::~WeakReference() {
	WITHACQUIRED(this->datalock);
	if(this->object)
		this->object->removeWeakRef(*this);
}

#endif
