/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_REFCOUNTER_DECLS_H
#define LIBMUTH_REFCOUNTER_DECLS_H

#include <set>
#include <stdexcept>
#include "lock.h"
#include "condition.h"

class WeakRefBase;
template<class RC> class WeakReference;

/**
 * Classes which require using reference counting can derive this class.
 */
class RefCounter {
	private:
		Lock datalock;
		Condition cond;
		int refcount;
		std::set<WeakRefBase*> notifier;
		void addWeakRef(WeakRefBase &wr);
		void removeWeakRef(WeakRefBase &wr);
		void notifyall();
		friend class WeakRefBase;
		template<class RC> friend class WeakReference;
	protected:
		/**
		 * Protected constructor as objects must be created using new,
		 * so use named constructor idiom.
		 */
		RefCounter();
	public:
		/**
		 * Increases reference count for this object.
		 */
		void incref();
		/**
		 * Decreases reference count for this object. The object may
		 * have disappeared afterwards.
		 */
		void decref();
		virtual ~RefCounter();
};

template<class RC> class WeakReference;

/**
 * This class is can be used like a pointer to RC objects. RC must be derived
 * from RefCounter.
 */
template<class RC> class Reference {
	private:
		RC *object;
		friend class WeakReference<RC>;
	public:
		/**
		 * Constructs a Reference from a RefCounter object created
		 * using new.
		 * @param obj is the object to reference
		 */
		Reference(RC *obj);
		/**
		 * This is a normal copy constructor needed for reference
		 * counting.
		 * @param other is the object to "copy"
		 */
		Reference(const Reference &other);
		/**
		 * Assigns a RefCounter reference to this Reference.
		 * @param obj is the object to reference
		 * @returns *this
		 */
		Reference<RC> &operator=(RC *obj);
		/**
		 * This is a normal assignment operator needed for reference
		 * counting.
		 * @param other is the object to "copy"
		 * @returns *this
		 */
		Reference<RC> &operator=(const Reference &other);
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the Reference to compare to
		 * @returns true if the references are equal, false otherwise
		 */
		bool operator==(const Reference &other) const;
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the Reference to compare to
		 * @returns true if the references are differ, false otherwise
		 */
		bool operator!=(const Reference &other) const;
		/**
		 * Returns the RefCounter derived object referenced.
		 * @returns the object referenced
		 */
		RC &operator*() const;
		/**
		 * Returns the RefCounter derived object referenced.
		 * @returns a pointer to the object referenced
		 */
		RC *operator->() const;
		/**
		 * This destructor is needed for reference counting.
		 */
		~Reference();
};

/**
 * Internal use only.
 */
class WeakRefBase {
		friend class RefCounter;
	protected:
		/**
		 * Internal use.
		 */
		mutable Lock datalock;
		/**
		 * Internal use for letting references die.
		 */
		virtual void notify() = 0;
		WeakRefBase();
		virtual ~WeakRefBase() {}
};

/**
 * This exception will be thrown when dereferencing a WeakReference whose
 * object has disappeared.
 */
class NullPointerException : public std::runtime_error {
	public:
		NullPointerException() : std::runtime_error("NullPointer") {}
};

/**
 * A weak reference is like a Reference, but the object disappears when there
 * are no more real References on it.
 */
template<class RC> class WeakReference : private WeakRefBase {
	private:
		RC *object;
		void notify();
		friend class Reference<RC>;
	public:
		/**
		 * Constructs a WeakReference that will throw a
		 * NullPointerException when dereferenced.
		 */
		WeakReference();
		/**
		 * Constructs a WeakReference by referencing the same object as
		 * another WeakReference other.
		 * @param other contains the reference to copy
		 */
		WeakReference(const WeakReference &other);
		/**
		 * Constructs a WeakReference by referencing the same object as
		 * a "real" Reference other.
		 * @param other contains the reference to copy
		 */
		WeakReference(const Reference<RC> &other);
		/**
		 * Constructs a WeakReference by referencing given obj. Note
		 * that there might not be "real" References so the referenced
		 * object might not be destroyed when this WeakReference dies.
		 * @param obj is a pointer to the object to be referenced
		 */
		WeakReference(RC *obj);
		/**
		 * The assignment operator is needed for reference counting.
		 * @param other is contains the reference to copy
		 * @returns *this
		 */
		WeakReference &operator=(const WeakReference &other);
		/**
		 * The assignment operator is needed for reference counting.
		 * @param other is contains the reference to copy
		 * @returns *this
		 */
		WeakReference &operator=(const Reference<RC> &other);
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the WeakReference to compare to
		 * @returns true if the references are equal, false otherwise
		 */
		bool operator==(const WeakReference &other) const;
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the Reference to compare to
		 * @returns true if the references are equal, false otherwise
		 */
		bool operator==(const Reference<RC> &other) const;
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the WeakReference to compare to
		 * @returns true if the references are differ, false otherwise
		 */
		bool operator!=(const WeakReference &other) const;
		/**
		 * Checks whether this and other reference the same object.
		 * @param other is the Reference to compare to
		 * @returns true if the references are differ, false otherwise
		 */
		bool operator!=(const Reference<RC> &other) const;
		/**
		 * Returns the RefCounter derived object referenced.
		 * @returns the object referenced
		 * @throws NullPointerException
		 */
		Reference<RC> operator*() const;
		/**
		 * This destructor is needed for reference counting.
		 */
		~WeakReference();
};

#endif
