/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef __WIN32__
#include "win32.h"
#endif
#include <map>
#include <queue>
#include <assert.h>
#ifndef __WIN32__
#include <sys/select.h>
#include <unistd.h>
#endif
#include <errno.h>
#include "selectscheduler.h"
#include "callback_decls.h"
#include "timeval.h"
#include "lock.h"
#include "globalqueue.h"

SelectScheduler::SelectScheduler(BlockingQueue<Microthread*> *q)
		: BaseScheduler(q ? q : &getglobalqueue()) {
#ifndef __WIN32__
	int p[2];
	if(::pipe(p) < 0)
		assert(0);
	this->rpipe = p[0];
	this->wpipe = p[1];
#endif
}

SelectScheduler &SelectScheduler::create(BlockingQueue<Microthread*> *q) {
	return *new SelectScheduler(q);
}

void SelectScheduler::notifyme() {
	if(this->notifylock.tryacquire()) {
#ifndef __WIN32__
		const int n(::write(this->wpipe, "\0", 1));
		assert(n == 1);
#endif
	}
}

void SelectScheduler::addReadCallback(Callback &cb, int fd) {
	assert(fd >= 0);
	{
		WITHACQUIRED(this->datalock);
#ifndef __WIN32__
		Callback *&rfd(this->rfds[fd]);
#else
		Callback *&rfd = this->rfds[fd];
#endif
		if(rfd != NULL) {
			assert(rfd->iscompleted());
			delete rfd;
		}
		rfd = &cb;
	}
	this->notifyme();
}

void SelectScheduler::addWriteCallback(Callback &cb, int fd) {
	assert(fd >= 0);
	{
		WITHACQUIRED(this->datalock);
#ifndef __WIN32__
		Callback *&wfd(this->wfds[fd]);
#else
		Callback *&wfd = this->wfds[fd];
#endif
		if(wfd != NULL) {
			assert(wfd->iscompleted());
			delete wfd;
		}
		wfd = &cb;
	}
	this->notifyme();
}

void SelectScheduler::addExceptCallback(Callback &cb, int fd) {
	assert(fd >= 0);
	{
		WITHACQUIRED(this->datalock);
#ifndef __WIN32__
		Callback *&efd(this->efds[fd]);
#else
		Callback *&efd = this->efds[fd];
#endif
		if(efd != NULL) {
			assert(efd->iscompleted());
			delete efd;
		}
		efd = &cb;
	}
	this->notifyme();
}

void SelectScheduler::addWaitCallback(Callback &cb, const timeval &tv) {
	bool notify(false);
	{
		WITHACQUIRED(this->datalock);
		if(this->waits.isempty()) {
			notify = true;
			this->waits.put(cb, tv);
		} else {
			const Timeval now(TimevalNow);
			const Timeval old(this->waits.delay(now));
			this->waits.put(cb, tv);
			if(old != this->waits.delay(now))
				notify = true;
		}
	}
	if(notify)
		this->notifyme();
}

void SelectScheduler::run() {
	std::queue<Callback*> callbackqueue;
	std::map<int, Callback*>::iterator i, j;
	int max, n;
	for(;;) {
		max = 0;
		/* Initialize rfdset, wfdset and max from this->rfdsched and
		 * this->wfdsched. */
		FD_ZERO(this->rfdset);
		FD_ZERO(this->wfdset);
		FD_ZERO(this->efdset);
		this->datalock.acquire();
		for(i = this->rfds.begin();i != this->rfds.end();) {
			assert(i->second != NULL);
			if(i->second->iscompleted()) {
				delete i->second;
				j = i;
				++i;
				this->rfds.erase(j);
			} else {
				FD_SET(i->first, this->rfdset);
				if(i->first >= max) max = i->first + 1;
				++i;
			}
		}
		for(i = this->wfds.begin();i != this->wfds.end();) {
			assert(i->second != NULL);
			if(i->second->iscompleted()) {
				delete i->second;
				j = i;
				++i;
				this->wfds.erase(j);
			} else {
				FD_SET(i->first, this->wfdset);
				if(i->first >= max) max = i->first + 1;
				++i;
			}
		}
		for(i = this->efds.begin();i != this->efds.end();) {
			assert(i->second != NULL);
			if(i->second->iscompleted()) {
				delete i->second;
				j = i;
				++i;
				this->efds.erase(j);
			} else {
				FD_SET(i->first, this->efdset);
				if(i->first >= max) max = i->first + 1;
				++i;
			}
		}

		/* Add the notification pipe to rfdset. */
#ifndef __WIN32__
		FD_SET(this->rpipe, this->rfdset);
		if(this->rpipe >= max) max = this->rpipe + 1;
#endif

		if(this->waits.cleanup()) {
			Timeval tv(this->waits.delay());
			this->datalock.release();
			do {
				errno = 0;
				n = ::select(max, this->rfdset, this->wfdset,
						this->efdset, &tv);
			} while(n < 0 && errno == EINTR);
		} else {
			this->datalock.release();
			do {
				errno = 0;
				n = ::select(max, this->rfdset, this->wfdset,
						this->efdset, NULL);
			} while(n < 0 && errno == EINTR);
		}

		/*
		 * No matter whether we succeed in getting the lock it will now
		 * be acquired, so no other microthread will acquire it and send
		 * stuff to the pipe. This will save us some iterations of this
		 * loop.
		 */
		this->notifylock.tryacquire();
		this->datalock.acquire();
		assert(n >= 0);

		/* This loop is run only once and enables the usage of break. */
		while(n) {
			/*
			 * This just makes us reenter the loop in order to visit
			 * changes to this->rfdsched and this->wfdsched.
			 */
#ifndef __WIN32__
			if(FD_ISSET(this->rpipe, this->rfdset)) {
				/* This is a write only buffer. */
				char devnull[1];
				const int r(::read(this->rpipe, devnull, 1));
				assert(r == 1);
				if(!--n) /* stop processing when done */
					break;

			}
#endif

			i = this->rfds.begin();
			while(i != this->rfds.end())
				if(FD_ISSET(i->first, this->rfdset)) {
					assert(i->second != NULL);
					/* Append to callbackqueue and run later
					 * as run might want to lock rfdlock. */
					callbackqueue.push(i->second);
					/* To remove *i, increment it and remove
					 * its previous value. This way the
					 * iterator stays valid. */
					j = i;
					++i;
					this->rfds.erase(j);
					if(!--n)
						break;
				} else
					++i;

			if(n <= 0) /* stop processing when done */
				break;

			i = this->wfds.begin();
			while(i != this->wfds.end())
				if(FD_ISSET(i->first, this->wfdset)) {
					assert(i->second != NULL);
					/* Append to callbackqueue and run later
					 * as run might want to lock rfdlock. */
					callbackqueue.push(i->second);
					/* Iterator stuff see above. */
					j = i;
					++i;
					this->wfds.erase(j);
					if(!--n)
						break;
				} else
					++i;

			if(n <= 0) /* stop processing when done */
				break;

			i = this->efds.begin();
			while(i != this->efds.end())
				if(FD_ISSET(i->first, this->efdset)) {
					assert(i->second != NULL);
					callbackqueue.push(i->second);
					j = i;
					++i;
					this->efds.erase(j);
					if(!--n)
						break;
				} else
					++i;

			break;
		}

		assert(n == 0);

		if(!this->waits.isempty()) {
			Timeval tv(TimevalNow);
			while(Callback *cb = this->waits.get(tv))
				callbackqueue.push(cb);
		}
		this->datalock.release();
		/* Empty the callbackqueue here. */
		while(!callbackqueue.empty()) {
			Callback *cb(callbackqueue.front());
			callbackqueue.pop();
			cb->run();
			delete cb;
		}

		while(!this->queue.isempty()) {
			this->scheduleme();
			try {
				this->delayme();
			} catch(...) {
				/* See comment below. */
				this->notifylock.release();
				throw;
			}
		}
		/*
		 * The lock was either acquired above or by another thread
		 * notifying us. So we release it to get notifies again.
		 */
		this->notifylock.release();
	}
}
