/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_SELECTSCHEDULER_H
#define LIBMUTH_SELECTSCHEDULER_H

#include <map>
#include <sys/types.h>
#ifndef __WIN32__
#include <sys/time.h>
#include <unistd.h>
#include <sys/select.h>
#else
#include "win32.h"
#endif
#include "basescheduler_decls.h"
#include "callback_decls.h"
#include "lock.h"
#include "timeval_decls.h"
#include "waitqueue.h"

/**
 * Reference implementation for schedulers.
 */
class SelectScheduler : public BaseScheduler {
	private:
		/*
		 * This lock locks all accesses to rfds, wfds and waits.
		 */
		Lock datalock;
		/*
		 * Whenever a Microthread wants to read or write from or to a
		 * filedescriptor it runs the scheduleReader or scheduleWriter
		 * method which will add the filedescriptor as key and the
		 * instance pointer as value to rfdsched or wfdsched. The pair
		 * will be discarded from the map when the filedescriptor
		 * becomes readable or writable.
		 */
		std::map<int, Callback*> rfds, wfds, efds;
		WaitQueue waits;

		Lock notifylock;
		int rpipe, wpipe;
		void notifyme();
		fd_set rfdset[1], wfdset[1], efdset[1];
	protected:
		/**
		 * Constructs a SelectScheduler running on queue q.
		 * @param q is the queue the scheduler needs to insert itself
		 */
		SelectScheduler(BlockingQueue<Microthread*> *q=NULL);
	public:
		/**
		 * Named constructor for SelectScheduler. The object is
		 * automatically deleted when run() returns.
		 * @param q is the queue the scheduler needs to insert itself
		 */
		static SelectScheduler &create(
				BlockingQueue<Microthread*> *q=NULL);
		void addReadCallback(Callback &cb, int fd);
		void addWriteCallback(Callback &cb, int fd);
		/**
		 * Works like addReadCallback for exceptions on file
		 * descriptors. See select 2 man page.
		 * @param cb is the Callback to be run
		 * @param fd is the filedescriptor to be monitored
		 */
		void addExceptCallback(Callback &cb, int fd);
		void addWaitCallback(Callback &cb, const timeval &tv);
		void run();
};

#endif
