/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_SUSPENDER_DECLS_H
#define LIBMUTH_SUSPENDER_DECLS_H

#include "refcounter_decls.h"
#include "lock.h"
#include "microthread.h"

/**
 * Suspends a Microthread and lets many others wake it. A Microthread gives
 * references to Suspender objects to other Microthread and invokes the
 * suspend() method. The first Microthread that invokes the wakeup(object)
 * method wakes the suspended Microthread. All other calls are ignored.
 */
template<class T> class Suspender : public RefCounter {
	private:
		mutable Lock lock;
		Microthread *microthread;
		bool wakened;
		T retval;
		/**
		 * Constructs a Suspender.
		 */
		Suspender() : microthread(NULL), wakened(false) {}
	public:
		/**
		 * This method wraps the constructor to make sure new is used.
		 * @returns the new Suspender object
		 */
		inline static Suspender &create() { return *new Suspender(); }
		/**
		 * Suspends the current Microthread until another Microthread
		 * invokes the wakeup() method.
		 * @returns the argument passed to wakeup()
		 */
		T suspend();
		/**
		 * Tries to wake the suspended Microthread with rv as return
		 * value.
		 * @param rv is returned by suspend() in the suspended
		 *           Microthread
		 * @returns true if this call really wakes the suspended
		 *           Microthread
		 */
		bool wakeup(const T &rv);
		/**
		 * Does the same as wakeup() with a direct context switch.
		 * @see wakeup
		 * @param rv is returned by suspend() in the suspended
		 *           Microthread
		 * @returns true if this call really wakes the suspended
		 *           Microthread
		 */
		bool cooperativewakeup(const T &rv);
		/**
		 * Checks whether a Microthread is currently suspended by this
		 * Suspender.
		 * @return true a Microthread called suspend and no other
		 *         called wakeup() yet
		 */
		inline bool issuspended() const;
		/**
		 * Checks whether wakeup() was called.
		 * @return true if any Microthread called wakeup()
		 */
		inline bool iswakened() const;
		/**
		 * Checks whether both suspend() and wakeup() are called.
		 * @return true if there was a suspended Microthread that
		 *         already woke up
		 */
		inline bool iscompleted() const;

};

#endif
