/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <fcntl.h>
#ifndef __WIN32__
#include <sys/socket.h>
#include <netinet/in.h>
#else
#include "win32.h"
#endif
#include <unistd.h>
#include "microthread.h"
#include "runner.h"
#include "iostream.h"
#include "scheduler.h"
#include "callback.h"
#include "cpudetect.h"

template<class Protocol> class Listener : public Microthread {
	protected:
		Scheduler &sched;
		int port;
	public:
		Listener(Scheduler &s, int p) : sched(s), port(p) {}
		void run();
};

template<class Protocol> void Listener<Protocol>::run() {
	struct sockaddr_in addr[1];
	addr->sin_family = AF_INET;
	addr->sin_port = htons(this->port);
	addr->sin_addr.s_addr = INADDR_ANY;
	const int fd(::socket(PF_INET, SOCK_STREAM, IPPROTO_TCP));
	assert(fd >= 0);
	int ret(1);
	ret = setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &ret, sizeof(ret));
	assert(ret >= 0);
	socklen_t addrlen(sizeof(*addr));
	ret = ::bind(fd, (struct sockaddr*)addr, addrlen);
	assert(ret >= 0);
	ret = ::listen(fd, 5);
	assert(ret >= 0);
	for(;;) {
		ret = this->sched.accept(fd, (struct sockaddr*)addr, &addrlen);
		assert(ret >= 0);
		(new Protocol(this->sched, ret))->scheduleme();
	}
}

class Forwarder : public Microthread {
	protected:
		BaseScheduler &sched;
		ScheduledOStream fd;
		char buff[4096];
	public:
		Forwarder(BaseScheduler &s, int d) : sched(s), fd(d, s) {}
		void run();
};

void Forwarder::run() {
	struct sockaddr_in addr[1];
	addr->sin_family = AF_INET;
	addr->sin_port = htons(22);
	static const uint8_t ipaddr[4] = { 127,0,0,1 };
	addr->sin_addr.s_addr = *(uint32_t*)ipaddr;
	const int sock(::socket(PF_INET, SOCK_STREAM, IPPROTO_TCP));
	int flags(fcntl(sock, F_GETFL));
	flags |= O_NONBLOCK;
	::fcntl(sock, F_SETFL, flags);
	this->sched.connect(sock, (struct sockaddr*)addr, sizeof(*addr));
	ScheduledOStream sfd(sock, this->sched);
	for(;;) {
		Reference<Suspender<bool> > sp(&Suspender<bool>::create());
		this->sched.addReadCallback(*new SuspendCallback<bool>(*sp, false), sfd);
		this->sched.addReadCallback(*new SuspendCallback<bool>(*sp, true), this->fd);
		if(sp->suspend()) {
			const int len(::read(this->fd, this->buff, 4096));
			if(len <= 0)
				break;
			sfd.write(this->buff, len);
			sfd.flush();
		} else {
			const int len(::read(sfd, this->buff, 4096));
			if(len <= 0)
				break;
			this->fd.write(this->buff, len);
			this->fd.flush();
		}
	}
	::close(this->fd);
	::close(sfd);
}

int main(void) {
	Scheduler &s(Scheduler::create());
	(new Listener<Forwarder>(s, 9999))->scheduleme();
	s.scheduleme();
	runnerLoop(detectCPUs());
	return 0;
}
