/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef USE_PTHREAD
#include <pthread.h>
#endif
#include <iostream>
#include <gmpxx.h>
#include "channel.h"
#include "microthread.h"
#include "runner.h"
#include "cpudetect.h"

template<class T>struct FactorJob {
	T number;
	T factor;
	bool operator<(const FactorJob &o) const { return number < o.number; }
	bool operator<=(const FactorJob &o) const { return number <= o.number; }
	bool operator==(const FactorJob &o) const { return number == o.number; }
	bool operator>=(const FactorJob &o) const { return number >= o.number; }
	bool operator>(const FactorJob &o) const { return number > o.number; }
	bool operator!=(const FactorJob &o) const { return number != o.number; }
};

template<class T> class Cracker : public Microthread {
	private:
		Channel<Channel<FactorJob<T>*>*> &reqchan;
		Channel<FactorJob<T>*> &reschan;
	public:
		Channel<FactorJob<T>*> jobchan;
		Cracker(Channel<Channel<FactorJob<T>*>*> &q,
				Channel<FactorJob<T>*> &r) : reqchan(q),
				reschan(r), jobchan(channelmanager) {}
		void run();
};

template<class T> void Cracker<T>::run() {
	for(;;) {
		reqchan.send(&this->jobchan);
		FactorJob<T> *fj(jobchan.receive());
		if(fj->number % 2 == 0)
			fj->factor = 2;
		else {
			T q(sqrt(fj->number));
			for(T n = 3; n <= q; n+= 2)
				if(fj->number % n == 0) {
					fj->factor = n;
					break;
				}
		}
		reschan.send(fj);
	}
}

template<class T> class Generator : public Microthread {
	private:
		T start;
	public:
		Channel<Channel<FactorJob<T>*>*> reqchan;
		Generator(T s) : start(s), reqchan(channelmanager) {}
		void run();
};

template<class T> void Generator<T>::run() {
	for(T n = this->start;;++n) {
		Channel<FactorJob<T>*> *c(this->reqchan.receive());
		FactorJob<T> *fj(new FactorJob<T>);
		fj->number = n;
		fj->factor = 0;
		c->send(fj);
	}
}

template<class T> class Processor : public Microthread {
	public:
		Channel<FactorJob<T>*> reschan;
		Processor() : reschan(channelmanager) {}
		void run();
};

template<class T> void Processor<T>::run() {
	for(;;) {
		FactorJob<T> *fj(reschan.receive());
		std::cout << fj->number.get_str() << ": " << fj->factor.get_str() << std::endl;
		delete fj;
	}
}

int main() {
	//const int cpus(detectCPUs());
	const int cpus(8);
	mpz_class s(1000000);
	s *= 1000000;
	Generator<mpz_class> *g(new Generator<mpz_class>(s));
	Processor<mpz_class> *p(new Processor<mpz_class>());
	g->scheduleme();
	p->scheduleme();
	for(int i=0;i<cpus;++i)
		(new Cracker<mpz_class>(g->reqchan, p->reschan))->scheduleme();
	runnerLoop(cpus);
}
