/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <assert.h>
#ifndef __WIN32__
#include <sys/socket.h>
#include <netinet/in.h>
#else
#include "win32.h"
#endif
#include "microthread.h"
#include "runner.h"
#include "scheduler.h"
#include "suspender.h"
#include "callback.h"
#include "iostream.h"
#include "refcounter.h"
#include "timeval.h"
#include "cpudetect.h"
#ifdef USE_PTHREAD
#include <pthread.h>
#endif

template<class Protocol> class Listener : public Microthread {
	protected:
		Scheduler &sched;
		int port;
	public:
		Listener(Scheduler &s, int p) : sched(s), port(p) {}
		void run();
};

template<class Protocol> void Listener<Protocol>::run() {
	struct sockaddr_in addr[1];
	addr->sin_family = AF_INET;
	addr->sin_port = htons(this->port);
	addr->sin_addr.s_addr = INADDR_ANY;
	const int fd(::socket(PF_INET, SOCK_STREAM, IPPROTO_TCP));
	assert(fd >= 0);
	int ret(1);
	ret = setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &ret, sizeof(ret));
	assert(ret >= 0);
	socklen_t addrlen(sizeof(*addr));
	ret = ::bind(fd, (struct sockaddr*)addr, addrlen);
	assert(ret >= 0);
	ret = ::listen(fd, 5);
	assert(ret >= 0);
	for(;;) {
		ret = this->sched.accept(fd, (struct sockaddr*)addr, &addrlen);
		assert(ret >= 0);
		(new Protocol(this->sched, ret))->scheduleme();
	}
}

class HTTPConnection : public Microthread {
	protected:
		BaseScheduler &sched;
		ScheduledOStream fd;
		char buff[4096];
	public:
		HTTPConnection(BaseScheduler &s, int d) : sched(s), fd(d, s) {}
		void run();
};

void HTTPConnection::run() {
	static const char html[] =
	"<html><head><title>foo</title></head><body><h1>foo</h1></body></html>";
	Timeval timeout(TimevalNow);
	timeout += (time_t)10;
	Reference<Suspender<bool> > sp(&Suspender<bool>::create());
	this->sched.addReadCallback(*new SuspendCallback<bool>(*sp, false), this->fd);
	this->sched.addWaitCallback(*new SuspendCallback<bool>(*sp, true), timeout);
	if(sp->suspend()) { /* timeout condition */
		::close(this->fd);
		return;
	}
	::read(this->fd, this->buff, 4096);
	this->fd << "HTTP/1.0 200 OK\r\n" <<
	"Content-length: " << sizeof(html)-1 << "\r\n" <<
	"Content-type: text/html\r\n" <<
	"\r\n" << html;
	this->fd.flush();
	::close(this->fd);
}

int main() {
	Scheduler &s(Scheduler::create());
	(new Listener<HTTPConnection>(s, 1080))->scheduleme();
	s.scheduleme();
	runnerLoop(detectCPUs());
	return 0;
}
