/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_TIMEVAL_H
#define LIBMUTH_TIMEVAL_H

#include <assert.h>
#include "timeval_decls.h"

/*
 * constructors
 */

inline Timeval::Timeval() {
}

inline Timeval::Timeval(const timeval &o) {
	assert(o.tv_usec < 1000000);
	tv_sec = o.tv_sec;
	tv_usec = o.tv_usec;
}

inline Timeval::Timeval(void*) {
#ifndef __WIN32__
	/* TODO: Return value ignored. */
	::gettimeofday(this, NULL);
#else
	struct _timeb tb;
	_ftime(&tb);
	tv_sec = tb.time;
	tv_usec = 1000*tb.millitm;
#endif
}

inline Timeval::Timeval(time_t sec) {
	tv_sec = sec;
	tv_usec = 0;
}

inline Timeval::Timeval(time_t sec, suseconds_t usec) {
	assert(usec < 1000000);
	tv_sec = sec;
	tv_usec = usec;
}

inline Timeval::Timeval(float sec) {
	tv_sec = (time_t)sec;
	sec -= (float) tv_sec;
	tv_usec = (suseconds_t)(sec * 1000000);
}

inline Timeval::Timeval(double sec) {
	tv_sec = (time_t)sec;
	sec -= (double) tv_sec;
	tv_usec = (suseconds_t)(sec * 1000000);
}

/*
 * operator=
 */

inline Timeval &Timeval::operator=(const timeval &o) {
	assert(o.tv_usec < 1000000);
	tv_sec = o.tv_sec;
	tv_usec = o.tv_usec;
	return *this;
}

inline Timeval &Timeval::operator=(time_t sec) {
	tv_sec = sec;
	tv_usec = 0;
	return *this;
}

inline Timeval &Timeval::operator=(float sec) {
	tv_sec = (time_t)sec;
	sec -= (float)tv_sec;
	tv_usec = (suseconds_t)(sec * 10000000);
	return *this;
}

inline Timeval &Timeval::operator=(double sec) {
	tv_sec = (time_t)sec;
	sec -= (double)tv_sec;
	tv_usec = (suseconds_t)(sec * 10000000);
	return *this;
}

/*
 * useful methods
 */
inline void Timeval::gettimeofday() {
#ifndef __WIN32__
	/* TODO: Return value ignored. */
	::gettimeofday(this, NULL);
#else
	struct _timeb tb;
	_ftime(&tb);
	tv_sec = tb.time;
	tv_usec = 1000*tb.millitm;
#endif
}

/*
 * casting operators
 */

inline Timeval::operator time_t() const {
	return tv_sec;
}

inline Timeval::operator float() const {
	return (float)tv_sec + 0.000001 * (float)tv_usec;
}

inline Timeval::operator double() const {
	return (double)tv_sec + 0.000001 * (double)tv_usec;
}

inline unsigned int Timeval::milliseconds() const {
	return (1000*(unsigned int)tv_sec)+(((unsigned int)tv_usec)/1000);
}

/*
 * comparison operators
 */

inline bool Timeval::operator>(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec > o.tv_sec || (tv_sec == o.tv_sec && tv_usec > o.tv_usec);
}

inline bool Timeval::operator>=(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec > o.tv_sec ||
			(tv_sec == o.tv_sec && tv_usec >= o.tv_usec);
}

inline bool Timeval::operator==(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec == o.tv_sec && tv_usec == o.tv_usec;
}

inline bool Timeval::operator<=(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec < o.tv_sec ||
			(tv_sec == o.tv_sec && tv_usec <= o.tv_usec);
}

inline bool Timeval::operator<(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec < o.tv_sec || (tv_sec == o.tv_sec && tv_usec < o.tv_usec);
}

inline bool Timeval::operator!=(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	return tv_sec != o.tv_sec && tv_usec != o.tv_usec;
}

/*
 * operator+
 */

inline Timeval Timeval::operator+(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	const suseconds_t t(tv_usec + o.tv_usec);
	if(t < 1000000)
		return Timeval(tv_sec + o.tv_sec, t);
	return Timeval(tv_sec + o.tv_sec + 1, t - 1000000);
}

inline Timeval Timeval::operator+(time_t sec) const {
	return Timeval(tv_sec + sec, tv_usec);
}

inline Timeval Timeval::operator+(float sec) const {
	/* TODO: more performance? */
	return *this + Timeval(sec);
}

inline Timeval Timeval::operator+(double sec) const {
	/* TODO: more performance? */
	return *this + Timeval(sec);
}

/*
 * operator+=
 */

inline Timeval &Timeval::operator+=(const timeval &o) {
	assert(o.tv_usec < 1000000);
	tv_sec += o.tv_sec;
	tv_usec += o.tv_usec;
	if(tv_usec >= 1000000) {
		++tv_sec;
		tv_usec -= 1000000;
	}
	return *this;
}

inline Timeval &Timeval::operator+=(time_t sec) {
	tv_sec += sec;
	return *this;
}

inline Timeval &Timeval::operator+=(float sec) {
	{
		const time_t t((time_t)sec);
		tv_sec += t;
		sec -= (float)t;
	}
	tv_usec += (suseconds_t)(sec * 1000000);
	while(tv_usec >= 1000000) {
		tv_usec -= 1000000;
		++tv_sec;
	}
	return *this;
}

inline Timeval &Timeval::operator+=(double sec) {
	{
		const time_t t((time_t)sec);
		tv_sec += t;
		sec -= (double)t;
	}
	tv_usec += (suseconds_t)(sec * 1000000);
	while(tv_usec >= 1000000) {
		tv_usec -= 1000000;
		++tv_sec;
	}
	return *this;
}

/*
 * operator-
 */

inline Timeval Timeval::operator-(const timeval &o) const {
	assert(o.tv_usec < 1000000);
	if(tv_usec >= o.tv_usec)
		return Timeval(tv_sec - o.tv_sec, tv_usec - o.tv_usec);
	return Timeval(tv_sec - o.tv_sec - 1, 1000000 - o.tv_usec + tv_usec);
}

inline Timeval Timeval::operator-(time_t sec) const {
	return Timeval(tv_sec - sec, tv_usec);
}

inline Timeval Timeval::operator-(float sec) const {
	/* TODO: more performance? */
	return *this - Timeval(sec);
}

inline Timeval Timeval::operator-(double sec) const {
	/* TODO: more performance? */
	return *this - Timeval(sec);
}

/*
 * operator-=
 */

inline Timeval &Timeval::operator-=(const timeval &o) {
	assert(o.tv_usec < 1000000);
	tv_sec -= o.tv_sec;
	if(tv_usec < o.tv_usec) {
		tv_usec += 10000000;
		tv_usec -= o.tv_usec;
		--tv_sec;
	} else
		tv_usec -= o.tv_usec;
	return *this;
}

inline Timeval &Timeval::operator-=(time_t sec) {
	tv_sec -= sec;
	return *this;
}

inline Timeval &Timeval::operator-=(float sec) {
	{
		time_t s((time_t)sec);
		tv_sec -= s;
		sec -= (float)s;
	}
	suseconds_t u((suseconds_t)(sec * 1000000));
	if(tv_usec < u) {
		tv_usec += 1000000;
		tv_usec -= u;
		--tv_sec;
	} else
		tv_usec -= u;
	return *this;
}

inline Timeval &Timeval::operator-=(double sec) {
	{
		time_t s((time_t)sec);
		tv_sec -= s;
		sec -= (double)s;
	}
	suseconds_t u((suseconds_t)(sec * 1000000));
	if(tv_usec < u) {
		tv_usec += 1000000;
		tv_usec -= u;
		--tv_sec;
	} else
		tv_usec -= u;
	return *this;
}

#endif
