/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_TIMEVAL_DECLS_H
#define LIBMUTH_TIMEVAL_DECLS_H

#include <time.h>
#ifndef __WIN32__
#include <sys/time.h>
#else
#include "win32.h"
#include <sys/timeb.h>
#include <sys/types.h>
#endif

#define TimevalNow ((void*)0)

/**
 * Timeval extends the struct timeval and adds useful methods.
 */
class Timeval : public timeval{
	public:
		/**
		 * Construct an uninitialized(!) Timeval.
		 */
		Timeval();
		/**
		 * Construct a Timeval by copying a timeval instance. As
		 * Timeval is derived from timeval this also covers the copy
		 * constructor.
		 * @param o is the timeval to copy.
		 */
		Timeval(const timeval &o);
		/**
		 * This constructor uses an ignored dummy parameter to
		 * initialize with gettimeofday(). The macro TimevalNow
		 * should be passed to use this constructor.
		 */
		Timeval(void*);
		/**
		 * Construct a Timeval by setting it to sec seconds.
		 * @param sec time in seconds to initialize with
		 */
		Timeval(time_t sec);
		/**
		 * Construct a Timeval by setting it to sec + 0.000001 * usec.
		 * @param sec time in seconds
		 * @param usec time in microseconds
		 */
		Timeval(time_t sec, suseconds_t usec);
		/**
		 * Construct a Timeval by setting it to sec.
		 * @param sec is a float in seconds
		 */
		Timeval(float sec);
		/**
		 * Construct a Timeval by setting it to sec.
		 * @param sec is a double in seconds
		 */
		Timeval(double sec);
		/**
		 * Assigns o to this Timeval. As Timeval is derived from
		 * timeval this also covers copying Timevals.
		 * @param o is the timeval to copy
		 * @return *this
		 */
		Timeval &operator=(const timeval &o);
		/**
		 * Assigns sec to this Timeval.
		 * @param sec is the time in seconds
		 * @return *this
		 */
		Timeval &operator=(time_t sec);
		/**
		 * Assigns sec to this Timeval.
		 * @param sec is the time in seconds
		 * @return *this
		 */
		Timeval &operator=(float sec);
		/**
		 * Assigns sec to this Timeval.
		 * @param sec is the time in seconds
		 * @return *this
		 */
		Timeval &operator=(double sec);
		/**
		 * Sets this to the current time by invoking gettimeofday.
		 */
		void gettimeofday();
		/**
		 * Converts Timeval to time_t by rounding down to seconds.
		 * @return rounded down seconds
		 */
		operator time_t() const;
		/**
		 * Converts Timeval to float. The result will be inexact.
		 * @return seconds of this Timeval as float
		 */
		operator float() const;
		/**
		 * Converts Timeval to double. The result will be inexact.
		 * @return seconds of this Timeval as double
		 */
		operator double() const;
		/**
		 * Converts Timeval to an unsigned int of milliseconds. It does
		 * not check for overflows.
		 * @returns the milliseconds of this Timeval as unsigned int
		 */
		unsigned int milliseconds() const;
		/**
		 * Checks whether this is larger than o.
		 * @param o the timeval to compare with
		 * @returns true if this > o
		 */
		bool operator>(const timeval &o) const;
		/**
		 * Checks whether this is larger than or equal to o.
		 * @param o the timeval to compare with
		 * @returns true if this >= o
		 */
		bool operator>=(const timeval &o) const;
		/**
		 * Checks whether this is equal to o. This will as with floats
		 * almost always give false.
		 * @param o the timeval to compare with
		 * @returns true if this == o
		 */
		bool operator==(const timeval &o) const;
		/**
		 * Checks whether this is smaller than or equal to o.
		 * @param o the timeval to compare with
		 * @returns true if this <= o
		 */
		bool operator<=(const timeval &o) const;
		/**
		 * Checks whether this is smaller than o.
		 * @param o the timeval to compare with
		 * @returns true if this < o
		 */
		bool operator<(const timeval &o) const;
		/**
		 * Checks whether this is not equal to o. This will as with
		 * floats almost always give true.
		 * @param o the timeval to compare with
		 * @returns true if this != o
		 */
		bool operator!=(const timeval &o) const;
		/**
		 * Returns the sum of this and o. Neither this nor o will be
		 * modified.
		 * @param o is the timeval to be added
		 * @return a new Timeval that equals to this + o
		 */
		Timeval operator+(const timeval &o) const;
		/**
		 * Returns the sum of this and sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be added
		 * @return a new Timeval that equals to this + sec
		 */
		Timeval operator+(time_t sec) const;
		/**
		 * Returns the sum of this and sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be added
		 * @return a new Timeval that equals to this + sec
		 */
		Timeval operator+(float sec) const;
		/**
		 * Returns the sum of this and sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be added
		 * @return a Timeval that equals to this + sec
		 */
		Timeval operator+(double sec) const;
		/**
		 * Adds o to this Timeval. This also works for Timeval as it is
		 * derived from timeval.
		 * @param o is the timeval to be added
		 * @return this modified object
		 */
		Timeval &operator+=(const timeval &o);
		/**
		 * Adds sec in seconds to this Timeval.
		 * @param sec is the time in seconds to add
		 * @return this modified object
		 */
		Timeval &operator+=(time_t sec);
		/**
		 * Adds sec in seconds to this Timeval.
		 * @param sec is the time in seconds to add
		 * @return this modified object
		 */
		Timeval &operator+=(float sec);
		/**
		 * Adds sec in seconds to this Timeval.
		 * @param sec is the time in seconds to add
		 * @return this modified object
		 */
		Timeval &operator+=(double sec);
		/**
		 * Returns the difference of this and o. The result is
		 * undefined if this < o. Neither this nor o will be modified.
		 * @param o is the timeval to be substracted
		 * @return a Timeval that equals to this + o
		 */
		Timeval operator-(const timeval &o) const;
		/**
		 * Returns the difference of this and sec. The result is
		 * undefined if this < sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be substracted
		 * @return a Timeval that equals to this + sec
		 */
		Timeval operator-(time_t sec) const;
		/**
		 * Returns the difference of this and sec. The result is
		 * undefined if this < sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be substracted
		 * @return a Timeval that equals to this + sec
		 */
		Timeval operator-(float sec) const;
		/**
		 * Returns the difference of this and sec. The result is
		 * undefined if this < sec. Neither this nor sec will be
		 * modified.
		 * @param sec is the time in seconds to be substracted
		 * @return a Timeval that equals to this + sec
		 */
		Timeval operator-(double sec) const;
		/**
		 * Substracts o from this Timeval. This also works for Timeval
		 * as it is derived from timeval.
		 * @param o is the timeval to be substracted
		 * @return this modified object
		 */
		Timeval &operator-=(const timeval &o);
		/**
		 * Substracts sec seconds from this Timeval.
		 * @param sec is the time in seconds to be substracted
		 * @return this modified object
		 */
		Timeval &operator-=(time_t sec);
		/**
		 * Substracts sec seconds from this Timeval.
		 * @param sec is the time in seconds to be substracted
		 * @return this modified object
		 */
		Timeval &operator-=(float sec);
		/**
		 * Substracts sec seconds from this Timeval.
		 * @param sec is the time in seconds to be substracted
		 * @return this modified object
		 */
		Timeval &operator-=(double sec);
};

#endif
