/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_UNITTEST_DECLS_H
#define LIBMUTH_UNITTEST_DECLS_H

#include <list>
#include <memory>
#include <stdexcept>
#include <string>

/**
 * A fixture is an environment to run test cases on. It is created for each case
 * prior to running it.
 */
class Fixture {
	public:
		virtual ~Fixture();
};

/**
 * A gatherer collects test results.
 */
class BaseGatherer {
	public:
		/**
		 * Create something that acts like this gatherer and can be
		 * deleted.
		 * @returns a deletable BaseGatherer
		 */
		virtual std::auto_ptr<BaseGatherer> clone()=0;
		/**
		 * Success reporting method.
		 * @param n is the name of the test that succeeded
		 */
		virtual void succeed(const std::string &n)=0;
		/**
		 * Failure reporting method.
		 * @param n is the name of the test that failed
		 * @param r is a reason for why the test failed
		 */
		virtual void fail(const std::string &n, const std::string &r)=0;
		/**
		 * Tells the gatherer that there are more reports to come
		 * @param n is the number of reports that are to arrive,
		 *          may be negative.
		 */
		virtual void pending(int n=1);
		/**
		 * Returns the gatherer that actually receives the result. This
		 * exists in order to avoid endless chains of ForwardGatherers.
		 * @returns the gatherer that this gatherer forwards results to
		 *          or this if this gatherer does not forward anything
		 */
		virtual BaseGatherer &forwardto();
		/**
		 * Returns what this gatherer prepends to the test name before
		 * forwarding the report.
		 * @returns the prefix or an empty string if this gatherer does
		 *          not forward anything
		 */
		virtual const std::string &prefixed() const;
		virtual ~BaseGatherer();
};

/**
 * Forwards all results to another gatherer prefixing the test names with a
 * common name.
 */
class ForwardGatherer : public BaseGatherer {
	private:
		BaseGatherer &gatherer;
		std::string prefix;
	public:
		std::auto_ptr<BaseGatherer> clone();
		/**
		 * Constructor.
		 * @param g is the gatherer to forward reports to
		 * @param p is the string to be prepended to test names
		 */
		ForwardGatherer(BaseGatherer &g, const std::string &p);
		void succeed(const std::string &n);
		void fail(const std::string &n, const std::string &r);
		void pending(int n);
		BaseGatherer &forwardto();
		const std::string &prefixed() const;
};

/**
 * Gathers statistics and prints test results.
 */
class Gatherer : public BaseGatherer {
	private:
		int succeeded;
		int failed;
	public:
		Gatherer();
		std::auto_ptr<BaseGatherer> clone();
		void succeed(const std::string &n);
		void fail(const std::string &n, const std::string &r);
		/**
		 * Print statistics on failed and succeeded tests.
		 */
		void stats();
		/**
		 * Returns the number of failed tests.
		 * @returns the number of failed tests
		 */
		int failed_tests();
};

/**
 * A test is a general interface to invoke something that may or may not fail.
 * It is implemented in TestCase and TestSuite.
 */
class BaseTest {
	protected:
		/**
		 * The name of the test.
		 */
		std::string name;
	public:
		/**
		 * BaseTest constructor.
		 * @param n ist the name of the test
		 */
		BaseTest(const std::string &n);
		/**
		 * Virtual copy constructor. This might be quite memory
		 * intensive as it actually copies deep.
		 * @returns an copy of this
		 */
		virtual BaseTest *clone() const=0;
		/**
		 * Runs the test reporting results to the provided gatherer.
		 * @param g is the gatherer results are reported to
		 */
		virtual void run(BaseGatherer &g)=0;
		virtual ~BaseTest();
};

/**
 * A testcase is a simple test consisting of a test function that may or may
 * not fail.
 */
template<class F> class TestCase : public BaseTest {
	private:
		void (F::*function)();
	public:
		/**
		 * TestCase constructor.
		 * @param n is the name of the test case
		 * @param func is the function to be invoked by testing
		 */
		TestCase(const std::string &n, void (F::*func)());
		/**
		 * Copy constructor.
		 * @param other is the TestCase to be copied
		 */
		TestCase(const TestCase &other);
		TestCase<F> *clone() const;
		void run(BaseGatherer &g);
};

/**
 * A suite groups a set of tests together and makes them available as one test.
 */
class TestSuite : public BaseTest {
	private:
		std::list<BaseTest*> tests;
	public:
		/**
		 * TestSuite constructor.
		 * @param n is the name of the suite
		 */
		TestSuite(const std::string &n);
		/**
		 * Copy constructor. This is a deep copy.
		 * @param o is the TestSuite to be copied
		 */
		TestSuite(const TestSuite &o);
		TestSuite *clone() const;
		/**
		 * Adds a given test to this suite.
		 * @param test is the test to add
		 */
		void addtest(BaseTest *test);
		void run(BaseGatherer &g);
		~TestSuite();
};

/**
 * This class is to be thrown from a test when the test fails.
 */
class Failure : public std::runtime_error {
	public:
		/**
		 * Construct a test failure with given message.
		 * @param r is the failure message
		 */
		Failure(const std::string &r);
};

/**
 * Runs the given test reporting results to the given gatherer. This function
 * might create a new thread or microthread in order to parallelize running
 * tests.
 * @param t is the test to be run
 * @param g is the gatherer results are to be reported to
 */
void unittest_run(BaseTest *t, BaseGatherer &g);

/**
 * Runs the test. A gatherer is created on which the test is run. After
 * printing statistics the program exits with the number of tests failed (at
 * most 255).
 * @param t is the test to be run
 * @returns never
 */
void unittest_main(std::auto_ptr<BaseTest> t);

#endif
