/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <algorithm>
#include <memory>
#include <assert.h>
#include <stdlib.h>
#include "unittest.h"
#include "microthread.h"
#include "channel.h"
#include "runner.h"
#include "cpudetect.h"

class TestResult {
	public:
		enum state_t { PENDING, SUCCESS, FAIL };
		state_t state;
		int count;
		std::string name;
		std::string message;
		TestResult(state_t s, int c=1, const std::string &n="",
				const std::string &m="")
				: state(s), count(c), name(n), message(m) {
			}
};

class GatherThread : public Microthread, public BaseGatherer {
	private:
		Gatherer gatherer;
		Channel<TestResult*> reschan;
		int pendingtests;
	public:
		GatherThread(int pend=0);
		std::auto_ptr<BaseGatherer> clone();
		void pending(int n);
		void succeed(const std::string &n);
		void fail(const std::string &n, const std::string &r);
		void run();

};

class TestRunner : public Microthread {
	private:
		std::auto_ptr<BaseTest> test;
		std::auto_ptr<BaseGatherer> gatherer;
	public:
		TestRunner(std::auto_ptr<BaseTest> o,
				std::auto_ptr<BaseGatherer> g);
		void run();
};

GatherThread::GatherThread(int pend)
		: reschan(channelmanager), pendingtests(pend) {
}

std::auto_ptr<BaseGatherer> GatherThread::clone() {
	return std::auto_ptr<BaseGatherer>(new ForwardGatherer(*this, ""));
}

void GatherThread::pending(int n) {
	this->reschan.send(new TestResult(TestResult::PENDING, n));
}

void GatherThread::succeed(const std::string &n) {
	this->reschan.send(new TestResult(TestResult::SUCCESS, 0,n));
}

void GatherThread::fail(const std::string &n, const std::string &r) {
	this->reschan.send(new TestResult(TestResult::FAIL, 0,n, r));
}

void GatherThread::run() {
	while(this->pendingtests > 0) {
		std::auto_ptr<TestResult> r(this->reschan.receive());
		switch(r->state) {
			case TestResult::PENDING:
				this->pendingtests += r->count;
				break;
			case TestResult::SUCCESS:
				if(!r->name.empty())
					this->gatherer.succeed(r->name);
				--this->pendingtests;
				break;
			case TestResult::FAIL:
				this->gatherer.fail(r->name, r->message);
				--this->pendingtests;
				break;
		}
	}
#ifdef NOEXITLEAK
	this->scheduleme();
	this->delayme();
#endif
	this->gatherer.stats();
	std::exit(std::min(this->gatherer.failed_tests(), 255));
}

TestRunner::TestRunner(std::auto_ptr<BaseTest> o, std::auto_ptr<BaseGatherer> g)
		: test(o), gatherer(g) {
}

void TestRunner::run() {
	this->test->run(*this->gatherer);
}

void unittest_run(BaseTest *t, BaseGatherer &g) {
	Microthread &m(getcurrentthread());
	m.scheduleme();
	m.swapto(*new TestRunner(std::auto_ptr<BaseTest>(t->clone()),
			g.clone()));
}

void unittest_main(std::auto_ptr<BaseTest> s) {
	GatherThread *g(new GatherThread(1));
	g->scheduleme();
	TestRunner *t(new TestRunner(s, g->clone()));
	t->scheduleme();
	runnerLoop(detectCPUs());
}
