/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include "waitqueue.h"
#include "timeval.h"

WaitQueue::Waiter::Waiter(Callback &cb, const timeval &tv) : Timeval(tv),
							     callback(cb) {
}

Timeval WaitQueue::Waiter::delay() const {
	const Timeval now(TimevalNow);
	if(*this < now)
		return (time_t)0;
	return *this - now;
}

Timeval WaitQueue::Waiter::delay(const timeval &now) const {
	if(*this < now)
		return (time_t)0;
	return *this - now;
}

inline bool WaitQueue::Waiter::iscompleted() const {
	return this->callback.iscompleted();
}

bool WaitQueue::isempty() const {
	return this->waits.empty();
}

Timeval WaitQueue::delay() const {
	assert(this->waits.empty() == false);
	return this->waits.top()->delay();
}

Timeval WaitQueue::delay(const timeval &now) const {
	assert(this->waits.empty() == false);
	return this->waits.top()->delay(now);
}

void WaitQueue::put(Callback &callback, const timeval &tv) {
	this->waits.push(new WaitQueue::Waiter(callback, tv));
}

Callback *WaitQueue::get() {
	if(this->waits.empty())
		return NULL;
	const Timeval now(TimevalNow);
	Waiter *const w(this->waits.top());
	if(w->iscompleted()) {
		this->waits.pop();
		delete &w->callback;
		delete w;
		return this->get(now);
	}
	if(*w > now)
		return NULL;
	this->waits.pop();
	Callback *const cb(&w->callback);
	delete w;
	return cb;
}

Callback *WaitQueue::get(const timeval &now) {
	if(this->waits.empty())
		return NULL;
	Waiter *const w(this->waits.top());
	if(w->iscompleted()) {
		this->waits.pop();
		delete &w->callback;
		delete w;
		return this->get(now);
	}
	if(*w > now)
		return NULL;
	this->waits.pop();
	Callback *const cb(&w->callback);
	delete w;
	return cb;
}

bool WaitQueue::cleanup() {
	while(!this->waits.empty()) {
		Waiter *const w(this->waits.top());
		if(w->iscompleted()) {
			this->waits.pop();
			delete &w->callback;
			delete w;
		} else
			return true;
	}
	return false;
}
