/*
 *  Copyright (C) 2006  Helmut Grohne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef LIBMUTH_WAITQUEUE_H
#define LIBMUTH_WAITQUEUE_H

#include <queue>
#include "callback_decls.h"
#include "timeval_decls.h"

/**
 * This class is a useful wrapper for timer events to be used in schedulers.
 */
class WaitQueue {
	private:
		class Waiter : public Timeval {
			public:
				Callback &callback;
				Waiter(Callback&, const timeval&);
				Timeval delay() const;
				Timeval delay(const timeval &now) const;
				bool iscompleted() const;
		};
		class WaiterComparator : public std::binary_function<Waiter*,
				Waiter*, bool> {
			public:
				bool operator()(Waiter *a, Waiter *b) {
						return *a > *b; }
		};

		std::priority_queue<Waiter*, std::vector<Waiter*>,
				WaiterComparator> waits;
	public:
		/**
		 * Checks whether the queue is empty. This may be run to find
		 * out whether delay() may be run.
		 * @return true if the queue is empty
		 * @see delay
		 */
		bool isempty() const;
		/**
		 * Returns how long to wait before next Callback is to be run.
		 * This must not be called on an empty WaitQueue.
		 * @return a Timeval containing the time to wait
		 */
		Timeval delay() const;
		/**
		 * Returns how long to wait before next Callback is to be run.
		 * This must not be called on an empty WaitQueue.
		 * @param now specifies the current time
		 * @return a Timeval containing the time to wait
		 */
		Timeval delay(const timeval &now) const;
		/**
		 * Puts a new Callback that is to be run at a given (absolute)
		 * time in the queue.
		 * @param callback is to be run at tv
		 * @param tv is the absolute time for running callback
		 */
		void put(Callback &callback, const timeval &tv);
		/**
		 * Gets a Callback that is to be run now from the WaitQueue.
		 * @return a Callback pointer if there are callbacks to be run
		 *         and NULL if the queue is empty or none is to be run
		 */
		Callback *get();
		/**
		 * Gets a Callback that is to be run now from the WaitQueue.
		 * @param now specifies the current time
		 * @return a Callback pointer if there are callbacks to be run
		 *         and NULL if the queue is empty or none is to be run
		 */
		Callback *get(const timeval &now);
		/**
		 * Removes completed callbacks from WaitQueue. Returns the same
		 * as !isempty(), i.e. did the cleanup leave elements?.
		 * @return !isempty()
		 * @see isempty
		 */
		bool cleanup();

};

#endif
